package TestUtilities;

/* Class Name           : TestExecution
* Version information   : Version 0.1
* Date                  : 07/02/1999
* Copyright Notice      : see below
* Edit history:
*    1 Mar 2000 Tom VanCourt    Added interfaces for doubles. Use
*               approximate equality tests. Removed float interface,
*               expecting float calls to be handled as doubles.
*   11 Feb 2000 Tom VanCourt    Added simpler interface. Since performance
*               isn't important, all data comparisons are converted to string
*               comparisons. That means there's really just one check method.
*    6 Feb 2000 Tom VanCourt    More cosmetic changes.
*   10 Jan 2000 Tom VanCourt    Change to meet coding standards.
*               Extended ints to longs for more versatility.
*               (Float->double extension risks changing semantics.)
*/


/*
    Copyright (C) 2000 Eric J. Braude and Thomas D. Van Court. 
 
    This program is the implementation of the case study specified in 
    "Software Engineering: an Object-Oriented Perspective," Wiley 2001,   
    by Eric J. Braude.

    The program is free software; you can redistribute it and/or modify it 
    under the terms of the GNU General Public License as published by the 
    Free Software Foundation; either version 2 of the License, or any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    For the GNU General Public License, see http://www.gnu.org/copyleft/gpl.txt 
    or write to the Free Software Foundation, Inc., 
    59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Eric Braude can be reached at ebraude@bu.edu and: Boston University,       
    MET Computer Science Department, 808 Commonwealth Avenue, Boston, MA 02215. 
    Tom Van Court can be reached at tvancour@hotmail.com
*/

import java.io.*;
import java.util.Date;

/** The TestExecution class.
*
* @author   Dr. Eric Braude
* @version  0.1
*/

public class TestExecution
{
    /** Standard message for indicating a defect */
    private static final String DEFECT_MSG =
        "\n======================= DEFECT ========================\n";

    /** Filename at which to generate output, by default. */
    private static final String DEFAULT_OUTPUT_FILE = "test_out.txt";

    /** File at which to generate output */
    private static PrintWriter testOutputS = new PrintWriter( System.out, true);

    /** True if testOutputS is based on stdout. */
    private static boolean usingStdoutS = true;

    /*----------------------------------------------------------------------*\
    ** Own interface
    \*----------------------------------------------------------------------*/

    /** Shut down the log file.
    */
    public static void closeTestLog()
    {
        if ( !usingStdoutS )            // Don't close stdout.
            testOutputS.close();

        testOutputS = null;             // Drop stale references.
        usingStdoutS = false;           // We're not using it.
    }

    /** Fetch the current log file, or open the default log file and fetch it.
    *
    * @return       File at which to geneerate test output.
    */
    private static PrintWriter getFile()
        { return getFile( DEFAULT_OUTPUT_FILE ); }

    /** Fetch the current log file, or open a named file and fetch it.
    * If that fails, type-cast stdout as a log file and use that.
    *
    * @param        outputFileNameP     Name of log file, if none is open yet.
    * @return       File at which to generate test output.
    */
    private static PrintWriter getFile(String outputFileNameP)
    {
        if ( testOutputS == null )                  // If file is open, use it.
            try {                                   // Open the requested file.
                testOutputS = new PrintWriter(
                    new FileOutputStream( outputFileNameP ), true );
                usingStdoutS = false;
            } catch( IOException ioex ) {           // Open stdout instead.
                System.out.println("!!! Failed to open " + outputFileNameP);
                testOutputS = new PrintWriter( System.out, true );
                usingStdoutS = true;
            }

        return testOutputS;
    }

    /** Open a file for test output. If that fails, use stdout.
    *
    * @param        outputFileNameP     Name of log file, if none is open yet.
    * @return       File at which to generate test output.
    */
    public static void openTestLog(String outputFileNameP)
    {
        closeTestLog();                     // Close previous log file
        getFile(outputFileNameP);           // Open a new log file.
    }

    /** Compares floats and prints each to a file.
    * Exact equality test is dicey for floats -- needs work.
    *
    * @param        aTestDescriptionP       A test description.
    * @param        aCorrectOutputP         The expected correct result.
    * @param        anActualOutputP         The actual result.
    * @return       <tt>true</tt> iff actual matched expected.
    */
    public static void println(String s)
        { getFile().println(s); }

    /** Compares strings and prints each to a file.
    *
    * @param        aTestDescriptionP       A test description.
    * @param        aCorrectOutputStringP   The expected correct result.
    * @param        anActualOutputStringP   The actual result.
    * @return       <tt>true</tt> iff actual matched expected.
    */
    public static boolean printReportToFile(
        String aTestDescriptionP,
        String aCorrectOutputStringP,
        String anActualOutputStringP )
    {
        boolean matchM = aCorrectOutputStringP.equals( anActualOutputStringP );
        PrintWriter aPrintWriterM = getFile();

        aPrintWriterM.println( "\n>>>>>" + aTestDescriptionP + "<<<<<" );
        aPrintWriterM.println( anActualOutputStringP + "<----Obtained\n"
            + aCorrectOutputStringP + "<---- Required" );
        if ( !matchM ) {
            aPrintWriterM.println( DEFECT_MSG );
            System.out.println( "DEFECT(S) FOUND: SEE OUTPUT FILE." );
        }
        return matchM;
    }

    private static final double
        absDoubleMismatch = 1.0e-6,
        relDoubleMismatch = 1.0e-6;

    /** Compares strings and prints each to a file.
    *
    * @param        aTestDescriptionP       A test description.
    * @param        aCorrectOutputP         The expected correct result.
    * @param        anActualOutputP         The actual result.
    * @return       <tt>true</tt> iff actual matched expected.
    */
    public static boolean printReportToFile(
        String aTestDescriptionP,
        double aCorrectOutputP,
        double anActualOutputP )
    {
        double largerM = Math.abs(
                Math.max( aCorrectOutputP, anActualOutputP ) ),
            smallerM = Math.abs( Math.min( aCorrectOutputP, anActualOutputP ) );
        boolean matchM = (smallerM == 0) ?
            (largerM < absDoubleMismatch) :
            (largerM - smallerM) < (largerM * relDoubleMismatch);

        PrintWriter aPrintWriterM = getFile();

        aPrintWriterM.println( "\n>>>>>" + aTestDescriptionP + "<<<<<" );
        aPrintWriterM.println( anActualOutputP + "<----Obtained\n"
            + aCorrectOutputP + "<---- Required" );
        if ( !matchM ) {
            aPrintWriterM.println( DEFECT_MSG );
            System.out.println( "DEFECT(S) FOUND: SEE OUTPUT FILE." );
        }
        return matchM;
    }

    /** Compares booleans and prints each to a file.
    *
    * @param        aTestDescriptionP       A test description.
    * @param        aCorrectOutputP         The expected correct result.
    * @param        anActualOutputP         The actual result.
    * @return       <tt>true</tt> iff actual matched expected.
    */
    public static boolean printReportToFile(
            String aTestDescriptionP,
            boolean aCorrectOutputP,
            boolean anActualOutputP )
    {
        return printReportToFile( aTestDescriptionP,
            aCorrectOutputP + "", anActualOutputP + "" );
    }

    /** Compares integers and prints each to a file.
    *
    * @param        aTestDescriptionP       A test description.
    * @param        aCorrectOutputP         The expected correct result.
    * @param        anActualOutputP         The actual result.
    * @return       <tt>true</tt> iff actual matched expected.
    */
    public static boolean printReportToFile(
            String aTestDescriptionP,
            long aCorrectOutputP,
            long anActualOutputP )
    {
        return printReportToFile( aTestDescriptionP,
            aCorrectOutputP + "", anActualOutputP + "" );
    }

    /** Create a very visible title block in the log file.
    *
    * @param        titleStringP            A test description.
    */
    public static void printTitle( String titleStringP )
    {
        println("*********************************************" +
            "*************************************************" +
            "******************");
        println("** " + titleStringP + " " + new Date() );
        println("*********************************************" +
            "*************************************************" +
            "******************");
    }

    /*----------------------------------------------------------------------*\
    ** Obsolete interfaces, maintained for compatibility.
    ** Don't use these in new coding.
    \*----------------------------------------------------------------------*/

    /** Obsolete name, preserved to for compatibility.
    *
    * @param        aFileWriterP            Destination for report output.
    * @param        aTestDescriptionP       A test description.
    * @param        aCorrectOutputP         The expected correct result.
    * @param        anActualOutputP         The actual result.
    */
    public static void printReportToFile( FileWriter aFileWriterP,
        String aTestDescriptionP,
        boolean aCorrectOutputP,
        boolean anActualOutputP )
    {
        reportToFile( aFileWriterP, aTestDescriptionP,
            aCorrectOutputP + "", anActualOutputP + "" );
    }

    /** Obsolete name, preserved to for compatibility.
    *
    * @param        aFileWriterP            Destination for report output.
    * @param        aTestDescriptionP       A test description.
    * @param        aCorrectOutputP         The expected correct result.
    * @param        anActualOutputP         The actual result.
    */
    public static void reportToFile( FileWriter aFileWriterP,
        String aTestDescriptionP,
        long aCorrectOutputP,
        long anActualOutputP )
    {
        reportToFile( aFileWriterP, aTestDescriptionP,
            aCorrectOutputP + "", anActualOutputP + "" );
    }

    /** Obsolete name, preserved to for compatibility.
    *
    * @param        aFileWriterP            Destination for report output.
    * @param        aTestDescriptionP       A test description.
    * @param        aCorrectOutputP         The expected correct result.
    * @param        anActualOutputP         The actual result.
    */
    public static void reportToFile( FileWriter aFileWriterP,
        String aTestDescriptionP,
        float aCorrectOutputP,
        float anActualOutputP )
    {
        reportToFile( aFileWriterP, aTestDescriptionP,
            aCorrectOutputP + "", anActualOutputP + "" );
    }

    /** Obsolete name, preserved to for compatibility.
    *
    * @param        aFileWriterP            Destination for report output.
    * @param        aTestDescriptionP       A test description.
    * @param        aCorrectOutputP         The expected correct result.
    * @param        anActualOutputP         The actual result.
    */
    public static void printReportToFile( FileWriter aFileWriterP,
        String aTestDescriptionP,
        float aCorrectOutputP,
        float anActualOutputP )
    {
        reportToFile( aFileWriterP, aTestDescriptionP,
            aCorrectOutputP + "", anActualOutputP + "" );
    }

    /** Obsolete name, preserved to for compatibility.
    *
    * @param        aFileWriterP            Destination for report output.
    * @param        aTestDescriptionP       A test description.
    * @param        aCorrectOutputP         The expected correct result.
    * @param        anActualOutputP         The actual result.
    */
    public static void reportToFile( FileWriter aFileWriterP,
        String aTestDescriptionP,
        String aCorrectOutputP,
        String anActualOutputP )
    {
        try {
            aFileWriterP.write( "\n\n>>>>>" + aTestDescriptionP + "<<<<<" );
            aFileWriterP.write( "\n" + aCorrectOutputP + "<----Obtained "
                + "\n" + anActualOutputP + "<---- Required" );
            if ( !aCorrectOutputP.equals( anActualOutputP ) ) {
                aFileWriterP.write( DEFECT_MSG );
                System.out.println( "DEFECT(S) FOUND: SEE OUTPUT FILE." );
            }
        } catch( IOException ioex) {
            System.out.println(ioex + "");
        }
    }

}
