package FrameworkRPG.GameEnvironment;

/* Class Name           : GameArea
* Version information   : Version 0.1
* Date                  : 07/02/1999
* Copyright Notice      : see below
* Edit history:
*   20 Feb 2000     Tom VanCourt    Started adding characters to areas.
*   18 Feb 2000     Tom VanCourt    Incorporated inspection comments.
*   06 Feb 2000     Tom VanCourt    Tighter name-setting checks.
*   08 Jan 2000     Tom VanCourt    Change to meet coding standards.
*/

/*
    Copyright (C) 2000 Eric J. Braude and Thomas D. Van Court. 
 
    This program is the implementation of the case study specified in 
    "Software Engineering: an Object-Oriented Perspective," Wiley 2001,   
    by Eric J. Braude.

    The program is free software; you can redistribute it and/or modify it 
    under the terms of the GNU General Public License as published by the 
    Free Software Foundation; either version 2 of the License, or any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    For the GNU General Public License, see http://www.gnu.org/copyleft/gpl.txt 
    or write to the Free Software Foundation, Inc., 
    59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Eric Braude can be reached at ebraude@bu.edu and: Boston University,       
    MET Computer Science Department, 808 Commonwealth Avenue, Boston, MA 02215. 
    Tom Van Court can be reached at tvancour@hotmail.com
*/

import FrameworkRPG.Debug;
import FrameworkRPG.Characters.*;
import TestUtilities.TestExecution;

/** Areas in which game characters interact.
* <p> Requirements:  SRS 3.2.AR
* <p> Design: SDD 3.1.3
* <p> Design issues: <ul>
* <li> This class accepts names composed of java.lang.Character.isDigit
*   and isLetter characters, not just A-Z, a-z, and 0-9. The former
*   is a much larger set, but is fully internationalized. The latter
*   is in the spec, but doesn't even handle European languages.
* </ul>
*
*
* @author   Dr. Eric Braude
* @version  0.1
*/

public abstract class GameArea
{
    /** Maximum length of area name <p> Req 3.2.AR.1.1 */
    private static final int MAX_NAME_LEN = 15;

    /** Default name. Note the special "-" character used in
    * the default name. That character is not accepted by the
    * setName method, so it's obvious that names with "-" were
    * assigned outside of normal execution.
    */
    private static String DEFAULT_NAME = "-AnonArea-";

    /** Assign a sequence number to each instance */
    private static int seqNumberS = 0;

    /** Sequence number for this instance */
    private final int seqNumberI = ++seqNumberS;

    /** Name of the game area. Initialized to some default.
    * The default name should never be reported -- it should always
    * be replaced by some assignment of the area's name.
    * <p> Requirement 3.2.AR.1.1
    */
    private String nameI = DEFAULT_NAME + seqNumberI;

    /** Implementation limit on number of characters in an area at once. */
    private static final int AREA_CHARACTERS =  5;

    /** Actual number of characters in an area at the moment. */
    protected int nAreaCharactersI = 0;

    /** List of characters in the area at the moment.
    * Non-null entries all go at low indices of the array,
    * in no specified order.
    * <p> Design statement: 3.1.4
    */
    protected GameCharacter[] areaCharactersI =
        new GameCharacter[AREA_CHARACTERS];

    /*----------------------------------------------------------------------*\
    ** Constructors
    \*----------------------------------------------------------------------*/

    /** Default constructor.
    */
    public GameArea()
    {
        this(null);
    }

    /** Construct a GamerArea object.
    * @param    areaNameP       Name assigned to the area.
    */
    public GameArea(String areaNameP)
    {
        super();
        setName(areaNameP);
    }

    /*----------------------------------------------------------------------*\
    ** Interface defined by this class
    \*----------------------------------------------------------------------*/

    /** Add a character to this area.
    * <p>Postcondition:     After normal completion of this method, the
    *                       character is listed as being in this area.
    * <p> Design statement: 3.1.4
    * @param    addedP      Character added to the area.
    */
    public void addCharacter( GameCharacter addedP )
    {
        areaCharactersI[ nAreaCharactersI++ ] = addedP;
    }

    /** Remove a character from this area.
    * <p>Postcondition:     After completing this method, the character is
    *                       not listed in this area.
    * <p> Design statement: 3.1.4
    * @param    deleteP     Character to remove from the area.
    */
    public void deleteCharacter( GameCharacter deleteP )
    {
        // Note that this removes possibly many instances of one
        // character from the list, ensuring the postcondition.
        int i = 0;

        while ( i < nAreaCharactersI )
            if ( deleteP == areaCharactersI[i] ) {  						// Remove from the list.
                areaCharactersI[i] = areaCharactersI[ --nAreaCharactersI ];
                areaCharactersI[nAreaCharactersI] = null;
            } else                                  						// Look farther down
                i += 1;                             						//  the list.
    }

    /** Report the number of characters in this area.
    * <p> Design statement: 3.1.4
    * @return   Number of characters in this area.
    */
    public int getNumCharacters()
        { return nAreaCharactersI; }

    /** Outputs the name of the game area.
    */
    protected void displayName()
        { Debug.log( this + " reported" ); }

    /** Accessor of the game area name.
    *
    * @return       The name of the game area.
    */
    public String getName()
        { return nameI; }

    /** Sets the area name to the specified name if length is within
    * the allowed length for a area name.  Otherwise the name is
    * truncated.  Inheritors should use this for setName( String ),
    * but not override.
    *
    * <p> Requirement:      SRS 3.2.AR.1.1 Area Name
    * @param        aNameP  Proposed area name.
    */
    protected final void setName( String aNameP )
    {
        boolean nameValidM = (aNameP != null) &&    // String is non-null,
            (aNameP.length() > 0);                  // and has >= 1 char;

        // Make sure all characters are valid: letters, digits, or spaces.
        for ( int i = 0; nameValidM && i < aNameP.length(); i++ ) {
            char nameChar = aNameP.charAt(i);
            nameValidM = Character.isLetter( nameChar ) ||
                Character.isDigit( nameChar ) ||
                ( nameChar == ' ' );
        }

        // Now assign this area's name.
        if ( !nameValidM ) {
            // Set name to default value and show this in system window.
            nameI = DEFAULT_NAME + seqNumberI;
            Debug.log( nameI + " selected by GameArea.setName()" );
        } else if ( aNameP.length() > MAX_NAME_LEN ) {
            // Set name to valid-length substring of the input.
            // Append "-" character to show system intervention in name.
            nameI = aNameP.substring( 0, MAX_NAME_LEN-1 ) + "-";
            Debug.log( "truncation in GameArea.setName()");
        } else
            nameI = new String( aNameP );
    }

    /*----------------------------------------------------------------------*\
    ** Test interface
    \*----------------------------------------------------------------------*/

    /** Test of methods
    *
    * @param        argsP   Command line arguments.
    */
    public static void main( String[] argsP )
    {
        boolean openedM = false;                    // Assume no log file.

        if ( argsP != null && argsP.length > 0 ) {  // If log file is specified,
            TestExecution.openTestLog( argsP[0] );  // then open it.
            openedM = true;
        }

        TestExecution.printTitle( "FrameworkRPG.GameEnvironment.GameArea" );

        GameArea theTestM = new GameArea() { };     // Start with default name
        String testNameM = "Test game area",
            longNameM = "Area name string too long to be accepted as valid";

        TestExecution.printReportToFile(
            "Game area default name (" + theTestM.getName() + ") prefix:",
            DEFAULT_NAME,
            theTestM.getName().substring( 0, DEFAULT_NAME.length() ) );

        theTestM.setName( testNameM );              // Basic name test.
        TestExecution.printReportToFile( "Game area name:",
            theTestM.getName(), testNameM);

        theTestM.setName( null );                   // Null string pointer.
        TestExecution.printReportToFile( "Null name string pointer: ",
            DEFAULT_NAME,
            theTestM.getName().substring( 0, DEFAULT_NAME.length() ) );

        theTestM.setName( longNameM );              // Over-long name.
        TestExecution.printReportToFile( "Over-long name stirng: ",
            longNameM.substring(0, MAX_NAME_LEN-1),
            theTestM.getName().substring( 0, MAX_NAME_LEN-1 ) );

        theTestM.setName( "" );                     // Null string.
        TestExecution.printReportToFile( "Zero-length name: ",
            DEFAULT_NAME,
            theTestM.getName().substring( 0, DEFAULT_NAME.length() ) );

        theTestM.setName( "dis-allowed!?" );        // Invalid characters.
        TestExecution.printReportToFile( "Invalid characters: ",
            DEFAULT_NAME,
            theTestM.getName().substring( 0, DEFAULT_NAME.length() ) );

        if ( openedM )                              // If we opened a file,
            TestExecution.closeTestLog();           // close it.
    }

    /*----------------------------------------------------------------------*\
    ** java.Object interface
    \*----------------------------------------------------------------------*/

    /** Generate a printable string summarizing the object.
    * Java defines this method as the type-coercion method for
    * converting this object to a String.
    *
    * @return       Printable string representing this object.
    */
    public String toString()
        { return getClass().getName() + " " + getName(); }

}
