package FrameworkRPG;

/* Class Name           : Debug
* Version information   : Version 0.1
* Date                  : 01/08/1999
* Copyright Notice      : see below
* Edit history:
*   24 Jul 2000     Tom VanCourt    Added message text to traceback
*   17 May 2000     Tom VanCourt    Added isDebugging.
*    9 Feb 2000     Tom VanCourt    Added assert.
*   08 Jan 2000     Tom VanCourt    Initial coding.
*/

/*
    Copyright (C) 2000 Eric J. Braude and Thomas D. Van Court. 
 
    This program is the implementation of the case study specified in 
    "Software Engineering: an Object-Oriented Perspective," Wiley 2001,   
    by Eric J. Braude.

    The program is free software; you can redistribute it and/or modify it 
    under the terms of the GNU General Public License as published by the 
    Free Software Foundation; either version 2 of the License, or any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    For the GNU General Public License, see http://www.gnu.org/copyleft/gpl.txt 
    or write to the Free Software Foundation, Inc., 
    59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Eric Braude can be reached at ebraude@bu.edu and: Boston University,       
    MET Computer Science Department, 808 Commonwealth Avenue, Boston, MA 02215. 
    Tom Van Court can be reached at tvancour@hotmail.com
*/

import java.io.*;

/** Debug utilities for role-playing game framework. 
*
* @author   Tom VanCourt
* @version  0.1
*/

public class Debug 
{
    /** Decide whether to throw exceptions on assertion failures. */
    private static boolean assertingS = true;
    
    /** Decide whether to freeze the program when requested. */
    private static boolean holdingS = true;
    
    /** Decide whether debug is active or not. 
    * For now, it's a composite of finer-grained debug flags.
    */
    private static boolean debuggingS = 
        assertingS || holdingS;
    
    /** Decide whether to create log entries for debug messages or not. */
    private static boolean loggingS = true;
    
    /** File to use for debug logging. */
    private static PrintStream dbLogS = System.out;
    
    /*----------------------------------------------------------------------*\
    ** Own interface
    \*----------------------------------------------------------------------*/
    
    /** Make sure the logical condition is true.
    * @param    testP       Logical condition being tested.
    */
    public static void assert( boolean testP )
        { assert( testP, null ); }
        
    /** Make sure the logical condition is true.
    * If false, throws an exception. This uses a RuntimeException,
    * since they don't need to be declared in a method's or constructor's
    * "throws" clause. That makes use of this method minimally invasive
    * in the code that invokes it.
    *
    * @param    testP       Logical condition being tested.
    * @param    msgP        String describing the condition.
    */
    public static void assert( boolean testP, String msgP )
    {
        if( testP )                     // Do nothing if true.
            return;
            
        if( msgP == null )              // Make sure there's a message.
            msgP = "logic failure";
           
        log(msgP);                      // Log entry for the traceback.
        traceback();                    // Log a call stack trace.
        
        if( assertingS )                // If enabled, throw an exception.
            throw new RuntimeException(msgP);
    }

    /** Close the current log file. */
    public static void close()
        { Debug.setLog( null ); }
        
    /** Assertion that's unconditionally false. */
    public static void fail()
        { assert(false); }
        
    /** Hold the program until user hits RETURN
    *
    * @param        s       Prompt string 
    */
    public static void hold()
        { Debug.hold( "Press ENTER to continue:" ); }
        
    /** Hold the program until user hits RETURN.
    *
    * @param        msgP    Prompt string 
    */
    public static void hold( String msgP )
    {
        System.out.println( msgP );      // System.out, not the debug log.
        
        if (holdingS)
            try { System.in.read();     // Wait for user input.
            } catch (Exception e1P) {
            }
        else                            // Say that we're not waiting.
            System.out.println( "  (continuing automatically)" );
    }
    
    /** Find out whether debugging is enabled.
    * @return       True iff debugging is enabled.
    */
    public static boolean isDebugging()
        { return debuggingS; }
    
    /** Create a log file entry for debugging.
    *
    * @param        msgP    String to be appended to log file.
    */
    public static void log( String msgP )
    {
        if ( loggingS )
            dbLogS.println(msgP);
    }
    
    /** Set the log file to use for output.
    *
    * @param    logNameP    String name of new output file.
    *                       If null, close the file and revert
    *                       to stdOut.
    */
    public static void setLog( String logNameP )
    {
        dbLogS.flush();
        if (dbLogS != System.out)           // If it's not stdout,
            dbLogS.close();                 //  close the output file.
            
        dbLogS = System.out;                // Use stdout until newlog opens.
        
        if (logNameP == null)               // Null file means stdout default.
            return;
                
        try {
            // The assignment never happens if the file-open fails.
            dbLogS = new PrintStream( new FileOutputStream(logNameP) );
        } catch (IOException ioex) {
            // Print file name, exception type, exception message. 
            dbLogS.println( "Problem opening "+ logNameP + ":"
                + ioex.getClass().getName() + " " + ioex.getMessage() );
        }
    }
    
    /** Generate log file entries that trace back the call stack.
    */
    public static void traceback( String msgP )
    {
        if( !loggingS )                     // Don't trace back when 
            return;                         // logging is turned off.
            
        if ( msgP == null )                 // Watch for null parameters.
            msgP = "";
            
        try {
            throw new Exception( msgP );    // Unconditionally throw an exception.
        } catch (Exception ex) {            // Unconditionally catch it.
            ex.printStackTrace( dbLogS );   // Dump the stack trace.
        }
    }
    
    public static void traceback() { traceback(""); }
}
