package FrameworkRPG.Characters;

/* Class Name           : GameCharacter
* Version information   : Version 0.1
* Date                  : 07/02/1999
* Copyright Notice      : see below
* Edit history:
*   11 Feb 2000     Tom VanCourt    Switch to simplified test interface.
*                                   main() conforms to test driver standard.
*   16 Jan 2000     Tom VanCourt    Added test cases.
*   08 Jan 2000     Tom VanCourt    Change to meet coding standards.
*/

/*
    Copyright (C) 2000 Eric J. Braude and Thomas D. Van Court. 
 
    This program is the implementation of the case study specified in 
    "Software Engineering: an Object-Oriented Perspective," Wiley 2001,   
    by Eric J. Braude.

    The program is free software; you can redistribute it and/or modify it 
    under the terms of the GNU General Public License as published by the 
    Free Software Foundation; either version 2 of the License, or any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    For the GNU General Public License, see http://www.gnu.org/copyleft/gpl.txt 
    or write to the Free Software Foundation, Inc., 
    59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Eric Braude can be reached at ebraude@bu.edu and: Boston University,       
    MET Computer Science Department, 808 Commonwealth Avenue, Boston, MA 02215. 
    Tom Van Court can be reached at tvancour@hotmail.com
*/

import java.io.*;
import java.util.*;

import FrameworkRPG.Debug;
import TestUtilities.*;

/** Character of role-playing games.
* <p>Requirements: SRS 3.2.EC Encounter Characters
* <p> Design: SDD 3.1.2, SDD 6.2.1
*
* <p>Design issues:<ul>
* <li> SRS 3.2.EC.1.1 is not followed exactly. Different name lengths
*   are allowed than in areas, and different characters (an i18n-clean
*   superset) are allowed than in the spec.
* </ul>
*
* @author   Dr. Eric Braude
* @version  0.1
*/

public abstract class GameCharacter 
{
    /** Base for default names. */
    protected static final String DEFAULT_NAME = "defaultName";
    
    /** Absolute limit on name length */
    private static final int ALLTIME_NAME_LIMIT = 100;
    
    /** Name of the game character. The default name
    * should always be reassigned by some call to setName().
    * <p> Requirement: SRS 3.2.EC.1.1
    */
    private String nameI = DEFAULT_NAME;

    /*----------------------------------------------------------------------*\
    ** Interface defined by this class
    \*----------------------------------------------------------------------*/

    /** Returns the maximum character length for a character name.
    * The represents an absolute limit, not to be over-ridden, 
    * so it's declared final.        
    *
    * @return       Maximum character name length.
    * @deprecated   N/A
    */
    public final int alltimeLimitOfNameLength() 
        { return ALLTIME_NAME_LIMIT; }

    /** Outputs the GameCharacter name to the debug log.
    */
    protected void display() 
        { Debug.log(this + " displayed"); }

    /** Returns the name of the GameCharacter object.
    *
    * @return       GameCharacter's name.
    */
    public String getName() 
        { return nameI; }

    /** Subclasses must declare limit on size of character names
    *
    * @return       The maximum length of a character name.
    */
    protected abstract int maxNumCharsInName();

    /** Sets the character name to the specified name if length is within
    * the allowed length for a character name.  Otherwise the name is
    * truncated.  Inheritors should use this for setName( String ),
    * but not override.
    * <p> Requirement: SRS 3.2.EC.1.1 Name of Encounter characters
    * <p> Design: SDD 6.2.1 GameCharacter class
    *
    * @param        aNameP  Proposed character name.
    */
    protected final void setName( String aNameP ) 
    {
        boolean nameValidM = (aNameP != null) &&    // String is non-null, 
            (aNameP.length() > 0) &&                // has >= 1 char,
            (maxNumCharsInName() > 0) &&            // and has valid checks.
            (maxNumCharsInName() <= alltimeLimitOfNameLength() );
            
        // Make sure all characters are valid: letters, digits, or spaces.
        for (int i = 0; nameValidM && i < aNameP.length(); i++) {
            char nameChar = aNameP.charAt(i);
            nameValidM =  Character.isLetter(nameChar) ||
                Character.isDigit(nameChar) ||
                (nameChar == ' ');
        }
            
        // Now assign this character's name. 
        if (!nameValidM) {
            // Set name to default value and show this in system window.
            nameI = new String( DEFAULT_NAME );
            Debug.log( "defaultName selected by GameCharacter.setName()" );
        } else if (aNameP.length() > maxNumCharsInName() ) {
            // Set name to valid-length substring of the input.
            nameI = new String( aNameP.substring( 0, maxNumCharsInName() ) );
            Debug.log( "truncation in GameCharacter.setName()");
        } else 
            nameI = new String( aNameP );
    }

    /*----------------------------------------------------------------------*\
    ** Test code
    \*----------------------------------------------------------------------*/

    /** To test this class.
    *   @param  argsP   One string: file name for test log. 
    *                   Use default log file if no name provided.
    */
    public static void main( String[] argsP ) 
    {
        boolean openedM = false;                    // Assume no log file.
        
        if ( argsP != null && argsP.length > 0 ) {  // If log file is specified, 
            TestExecution.openTestLog( argsP[0] );  // then open it.
            openedM = true;
        }
        
        TestExecution.printTitle(                   // Title the log entries.
            "FrameworkRPG.Characters.GameCharacter" );
            
        testForGameCharacterClass();
    
        if ( openedM )                              // If we opened a file,
            TestExecution.closeTestLog();           // close it. 
    }
    
    /** Tests all the methods of this class
    *
    * @exception    IOException     if there are problems accessing aDestinationP
    */
    private static void testForGameCharacterClass()
    {
        GameCharacter cM  = new GameCharacter() {              
            // Defines a name length
            public int maxNumCharsInName() { return 20; }
        };
        
        /* Because this is an abstract class */
        System.out.println(
            "Test for GameCharacter class with maxNumCharsInName() = " 
            + cM.maxNumCharsInName() );

        /* Tests for SetName() */
        
        /* SetName() 1. Equivalence partitioning (one partition): 
        * nominal, legal name 
        */
        cM.setName( "Harry" );
        TestExecution.printReportToFile(
            "SetName Test 1.1: Non-null string", cM.nameI, "Harry");
        cM.setName( "has blanks" );
        TestExecution.printReportToFile(
            "SetName Test 1.2: blanks", cM.nameI, "has blanks" );
        cM.setName( "has1or2digits" );
        TestExecution.printReportToFile(
            "SetName Test 1.3: digits", cM.nameI, "has1or2digits" );

        /* SetName() 2. Boundary value analysis */
        cM.setName( "X" );
        TestExecution.printReportToFile(
            "SetName Test 2.1: Non-null string", cM.nameI, "X" );
        cM.setName( "12345678901234567890" );
        TestExecution.printReportToFile(
            "SetName Test 2.2: Non-null string", cM.nameI, 
            "12345678901234567890" );

        /* SetName() 3. Illegal values */
        cM.setName( null );
        TestExecution.printReportToFile(
            "SetName Test 3.1: Null string", cM.nameI, DEFAULT_NAME );
        cM.setName( "1234567890123456789012345678901234567890" );
        TestExecution.printReportToFile(
            "SetName test 3.2: very long name", cM.nameI, 
            "12345678901234567890" );
        cM.setName( "" );
        TestExecution.printReportToFile(
            "SetName Test 3.3: empty string", cM.nameI, DEFAULT_NAME );
        cM.setName( "'hyphen-ated.'");
        TestExecution.printReportToFile(
            "SetName Test 3.4: non letter/digit", cM.nameI, DEFAULT_NAME );

        /*
         * Statement coverage: verified that every statement of setName() 
         *  is covered by one or more of the tests above
         * Decision coverage: verified that every "true" and "false" 
         *  branch has been executed by the tests above
         */

        /* Conclude */
        System.out.println( "\nTest of GameCharacter class concluded." );
    }
    
    /*----------------------------------------------------------------------*\
    ** java.Object interface
    \*----------------------------------------------------------------------*/
    
    /** Generate a printable string summarizing the object.
    * Java defines this method as the type-coercion method for
    * converting this object to a String.
    *
    * @return       Printable string representing this object.
    */
    public String toString()
        { return getClass().getName() + ' ' + getName(); }
    
}
