package Encounter;

/* Class Name           : GameMain
* Version information   : Version 0.1
* Date                  : 3/8/2000
* Copyright Notice      : see below
* Edit history:
*   31 May 2000 Tom VanCourt    Simplified initialization.
*   14 May 2000 Tom VanCourt    Moved to default package.
*   10 Apr 2000 Tom VanCourt    Added Frame param to setDisplay
*    3 Apr 2000 Tom VanCourt    Rework after review: bug fix in
*                               'end game' button handler.
*   20 Mar 2000 Tom VanCourt    Connect UI buttons to game logic.
*   10 Mar 2000 Tom VanCourt    Incorporate review comments. 
*    8 Mar 2000 Tom VanCourt    Initial coding
*/

/*
    Copyright (C) 2000 Eric J. Braude and Thomas D. Van Court. 
 
    This program is the implementation of the case study specified in 
    "Software Engineering: an Object-Oriented Perspective," Wiley 2001,   
    by Eric J. Braude.

    The program is free software; you can redistribute it and/or modify it 
    under the terms of the GNU General Public License as published by the 
    Free Software Foundation; either version 2 of the License, or any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    For the GNU General Public License, see http://www.gnu.org/copyleft/gpl.txt 
    or write to the Free Software Foundation, Inc., 
    59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Eric Braude can be reached at ebraude@bu.edu and: Boston University,       
    MET Computer Science Department, 808 Commonwealth Avenue, Boston, MA 02215. 
    Tom Van Court can be reached at tvancour@hotmail.com
*/

import java.awt.*;
import java.awt.event.*;

import Encounter.EncounterGame.*;
import FrameworkRPG.RolePlayingGame.*;
import FrameworkRPG.Debug;

/** Top-level entry point for the Encounter game.
* This class just defines the UI layout and acts as interface to the GUI 
* system. This  does not contain any of the logic of playing the game. 
* This does, however, instantiate the game just once.
* <p> This class couples the game logic to the Java application execution 
* environment, loosely following the Object Adapter design patterns ("Design
* Patterns", Gamma et. al.) Depending on how you look at it, this class 
* takes the 'Adapter' role in that pattern, EncounterGame takes the 'Adaptee'
* role, and the execution environment takes the 'Client' role. The role of 
* 'Target' can not be expressed exactly in Java syntax, but is the interface
* that defines <tt>main( String[] )<\tt> as the application's entry point. 
* One could also describe java.awt.Frame and other UI objects as the Targets
* for connecting the game to the AWT user interface.
* <p> Requirements: SRS 3.1.1 - This class defines the main screen layout,
*   including play area, control buttons, and thumbnail map. 
* <br> SRS 3.2.1.1  <I>Initialize</I>use case. 
* <br> SRS 3.2.AR.1.2 Area Image
* <br> SRS 3.2.AR.1.4 Action buttons
* <br> SRS 3.2.AR.4.3-6 are handled, at least in part, by ActionListener
*   objects defined in this class.
* <p> Design: SDD 3.2 Concurrent process description and SDD 5.2.1 Player
*   Character Movement Process. This class doesn't
*   actually implement a thread of control, but this is as good a place
*   as any to note the system's default UI thread, where much of the
*   game's activity occurs.
*
* <p> Design issues:<ul>
* <li> This class is not metioned at all in the design document. 
*   It was introduced to isolate the logic of the game from the environment
*   in which Java runs an application.
* <li> SRS 3.2.AR.1.2 talks about the area image filling the screen. This
*   is a window-oriented implementation. Even so, the area image only fills
*   most of the window, not 100% of the window. Small bits of real estate
*   are committed to the control buttons and thumbnail map.
* </ul>
* @author   Tom VanCourt
* @version  0.1
*/

public class GameMain extends Frame implements ActionListener
{
    private static final int
        THUMBNAIL_WIDTH = 125,                  // Thumbnail map size
        THUMBNAIL_HEIGHT = 75;
        
    /** Area image fills most of the window <p> Requirement SRS 3.2.AR.1.2 */
    static final int GAME_AREA_WIDTH = 600;
            
    /** Area image fills most of the window <p> Requirement SRS 3.2.AR.1.2 */
    static final int GAME_AREA_HEIGHT = 400;
        
    /** Game object: the whole thing. 
    * There's only one, per requirement 3.2.EG.2.1
    */
    private EncounterGame gameI;
    
    /*----------------------------------------------------------------------*\
    ** Constructors
    \*----------------------------------------------------------------------*/

    /** Create the new frame (UI/display) object. Very little happens here,
    * since it's too early for the UI manager's internal event processing 
    * to work right. That means we have to add components into the frame 
    * later, once the frame object is fully instantiated.
    */
    private GameMain()
    {
        super("Encounter Game");
        gameI = EncounterGame.getTheEncounterGame();
    }
    
    /*----------------------------------------------------------------------*\
    ** java.awt.Frame interface
    \*----------------------------------------------------------------------*/
    
    /** Set up the main UI window. At the time this is called, well after
    * object instantiation, the window and its internal event processing  
    * have been created, so this setup sequence can run.
    * <p>Requirements:  3.1.1 shows the game's over-all screen layout, and
    *       3.2.AR.1.4, placement of control buttons.
    */
    private void createGameUI()
    {
        Component                                   // Build UI objects.
            smallMapM = new Canvas() {              // Place for thumbnail
                public void paint(Graphics g) {     // Repainting this canvas
                    gameI.drawThumbnail( this );    //  the thumbnail map.
                    }
                },
            gameAreaM = new Canvas() {              // Place for area display
                public void update(Graphics g){     // Don't blank
                    paint( g ); 
                    }
                public void paint(Graphics g) {     // Repainting this canvas
                    gameI.drawPlayerArea( this );   //  the playing area.
                    }
                },
            controlItemM[] = {                      // Row below play area.
                makeButton( "Get Status", new ActionListener() {
                    public void actionPerformed( ActionEvent eP ) 
                        { gameI.showStatus(); }     // Show game status
                    } ),
                makeButton( "Set Qualities", new ActionListener() {
                    public void actionPerformed( ActionEvent eP ) 
                        { gameI.setQualities(); }   // Set player qualities
                    } ),
                makeButton( "End Game", this),      // Destroy frame and the game
                smallMapM,                          // Thumbnail map.
            };
        
        smallMapM.setSize( THUMBNAIL_WIDTH,         // Size of the thumbnail. 
            THUMBNAIL_HEIGHT );
        
        gameAreaM.setSize( GAME_AREA_WIDTH,         // Size of the play area. 
            GAME_AREA_HEIGHT );
            
        gameAreaM.addMouseListener(                 // Respond to a mouse click.
            new RPGMouseEventListener( gameI ) );
            
        gameI.setDisplay( this, gameAreaM );        // Game may update screen.
            
        Panel controlRow = new Panel();             // Bottom row of screen.
        for (int i= 0; i < controlItemM.length; i++)
            controlRow.add( controlItemM[i] );
            
        setLayout( new BorderLayout() );            // Main window layout.
        add( controlRow, BorderLayout.SOUTH );      // Control area at bottom.
        add( gameAreaM, BorderLayout.CENTER );      // Playing area
    }
        
    /*----------------------------------------------------------------------*\
    ** Application entry point
    \*----------------------------------------------------------------------*/

    /** Start the application. 
    * @param    argv        Command line parameters.
    */
    public static void main( String[] argv )
    {
        try {
            // Create the main application window.
            final GameMain gameFrameM = new GameMain();
            gameFrameM.addWindowListener( new WindowAdapter() {
                public void windowClosed( WindowEvent e ) 
                    { System.exit(0); }
                public void windowClosing( WindowEvent e ) 
                    { gameFrameM.dispose(); }
                }
            );
            
            // Set frame to a workable size. The size given here is only
            // approximate, since the frame can be resized and the layout
            // manager can adapt the content over a wide range of sizes.
            // Add in approximate amounts for window frames, etc.
            gameFrameM.resize( GAME_AREA_WIDTH + 25, 
                GAME_AREA_HEIGHT + THUMBNAIL_HEIGHT + 25 );
            gameFrameM.createGameUI();      // Set up the user interface. 
            gameFrameM.show();              // Display the window.
            gameFrameM.validate();          // Display the window contents.
            gameFrameM.toFront();           // Start in front of other windows.
        } catch ( Throwable thrownException ) {
            // Backstop exception handler: Catch any exceptions not already 
            // handled elsewhere and try to get debug information from them.
            // RuntimeException objects, in particular, can be thrown by
            // any method without having been declared in a 'throws' clause.
            System.err.println( thrownException.getMessage() ); 
            thrownException.printStackTrace( System.err );
        } 
    }
  
    /*----------------------------------------------------------------------*\
    ** Interface defined by this class
    \*----------------------------------------------------------------------*/
    
    /** Create a button with a specified event handler. 
    * @param    nameP       Printable caption shown on the button.
    * @param    actionP     Button-press handler 
    * @return               The new button object.
    */
    private Button makeButton( String nameP, ActionListener actionP )
    {
        Button buttonM = new Button( nameP );
        buttonM.addActionListener( actionP );
        return buttonM;
    }
    
    /*----------------------------------------------------------------------*\
    ** ActionListener interface 
    \*----------------------------------------------------------------------*/
    
    /** Destroy the frame and the game. This gets connected to the 
    * 'end game' button.
    */
    public void actionPerformed(ActionEvent e)
    {
        // Lot more shutdown logic could go here. 
        
        dispose();                              // Release this window.
        System.exit( 0 );                       // End the program.
    }
}
