package Encounter.EncounterGame;

/* Class Name           : Waiting
* Version information   : Version 0.1
* Date                  : 07/02/1999
* Copyright Notice      : see below
* Edit history:
*   20 May 2000     Tom VanCourt    Removed logic for changing game state
*                                   when player goes to new area.
*   14 May 2000     Tom VanCourt    Fixed display of new area after moving
*   13 May 2000     Tom VanCourt    Transition to the engagement state if
*                                   character movement enters an inhabited area.
*    4 May 2000     Tom VanCourt    Correct state transitions for
*                                   displaying status and changing qualities.
*   10 Apr 2000     Tom VanCourt    Filled in showStatus, setQualities
*   30 Mar 2000     Tom VanCour     Subclassed from EncounterGameState
*   13 Mar 2000     Tom VanCourt    Added character motion events
*   08 Jan 2000     Tom VanCourt    Change to meet coding standards.
*/

/*
    Copyright (C) 2000 Eric J. Braude and Thomas D. Van Court. 
 
    This program is the implementation of the case study specified in 
    "Software Engineering: an Object-Oriented Perspective," Wiley 2001,   
    by Eric J. Braude.

    The program is free software; you can redistribute it and/or modify it 
    under the terms of the GNU General Public License as published by the 
    Free Software Foundation; either version 2 of the License, or any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    For the GNU General Public License, see http://www.gnu.org/copyleft/gpl.txt 
    or write to the Free Software Foundation, Inc., 
    59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Eric Braude can be reached at ebraude@bu.edu and: Boston University,       
    MET Computer Science Department, 808 Commonwealth Avenue, Boston, MA 02215. 
    Tom Van Court can be reached at tvancour@hotmail.com
*/

import java.awt.*;
import java.awt.event.*;
import java.io.*;

import Encounter.EncounterCharacters.*;
import Encounter.EncounterGame.EncounterGameDisplays.*;
import Encounter.EncounterEnvironment.*;
import FrameworkRPG.Debug;
import FrameworkRPG.RolePlayingGame.*;

/** State of the Encounter game in which nothing is happening,
* and the player does not ask to do anything special.
* <p> Requirements: See the state diagram, section 2.1.1. 
* <br> Handle state-dependent parts of use case SRS 2.2.2. 
* <br> Handles the part of SRS 2.2.3 initiated by the player character.
* <br> SRS 3.2.AR.4.3, 3.2.AR.4.4, and 3.2.AR.4.6 are state-dependent
*   behaviors with handlers in this class.
* <p> Design: SDD 3.4 State Model Decomposition.
* <br> SDD 6.1.1 Detailed design of <i>RolePlayingGame</i> package
*
* <p> Design issues:<ul>
* <li> This class references an EncounterGame object; that reference is
*   not shown in the design UML. Reference to the EncounterGame was added
*   so this state object could access the current Area, for handling mouse
*   events, and so this state object could change the game as a whole to a
*   different state. This implementation chose to pass the game object to
*   this state via explicit parameter passing, rather than through
*   undisciplined access to static global data.
* <li> Event handling differs significantly from the model in SRS 2.2.2
* </ul>
* @author   Dr. Eric Braude
* @version  0.1
*/

class Waiting extends EncounterGameState
{
    /*----------------------------------------------------------------------*\
    ** Constructors
    \*----------------------------------------------------------------------*/

    /** Create a new state handler, waiting for game events to occur.
    * @param    gameM       Game context in which player waits.
    */
    public Waiting ( EncounterGame gameM )
    {
        super( gameM );
    }

    /*----------------------------------------------------------------------*\
    ** FrameworkRPG.GameEnvironment.GameState interface
    \*----------------------------------------------------------------------*/

    /** Reverts to event handling in the particular state.
    * <p> Requirements:     SRS 2.1.1, 3.2.1.2,
    *   <br> SRS 3.2.CO.4.1 Moving a character through a connection
    * <p> Use case:         SRS 2.2.2
    * <p> Design:           SDD 3.4
    * @param    eventP  The current event being handled.
    */
    public void handleEvent( AWTEvent eventP )
    {
        super.handleEvent( eventP );                // Default handling.

        MouseEvent mouseEventM =                    // Treat the event as
            (MouseEvent) eventP;                    // a mouse event.

        Area playerArea = gameI.getPlayerArea();    // Find the player's area.

        Component drawAreaM =                       // Get the screen area
            mouseEventM.getComponent();             // where event happened.

        Hyperlink areaLinkM =                       // Find hyperlink in area,
            playerArea.getHyperlink( drawAreaM,     //  in that screen display,
                mouseEventM.getX(),                 //  closest to the mouse.
                mouseEventM.getY() );

        if ( areaLinkM != null ) {                  // No link was nearby.
            Area goesTo =                           // Pass through connection.
                areaLinkM.getOtherArea( playerArea );

            gameI.movePlayerTo( goesTo );           // Move the player.
                
            gameI.redisplay( goesTo );              // Display the new area.
        }
    }

    /*----------------------------------------------------------------------*\
    ** EncounterGameState interface -- state-dependent handling of
    **      UI button press events.
    \*----------------------------------------------------------------------*/

    /** Handle the user interface event that occurs on a request to
    * change the player's qualities.
    * <p> Requirements:     SRS 3.2.PQ
    */
    public void setQualities()
    {
        gameI.setState( new SettingQualities( gameI ) );
    }

    /** Handle the user interface event that occurs on a request to display
    * the player's game status.
    * <p> Requirements:     SRS 3.2.EC.3.2
    */
    public void showStatus()
    {
        gameI.setState( new ReportingEncounter( gameI ) );
    }
}
