package Encounter.EncounterGame;

/* Class Name           : Engaging
* Version information   : Version 0.1
* Date                  : 07/02/1999
* Copyright Notice      : see below
* Edit history:
*   20 May 2000     Tom VanCourt    Fetch environment from game object.
*                                   Changed logic of jumping character
*                                   to another area and watching for
*                                   re-engagements.
*   12 May 2000     Tom VanCourt    Filled in body
*   08 Jan 2000     Tom VanCourt    Change to meet coding standards.
*/

/*
    Copyright (C) 2000 Eric J. Braude and Thomas D. Van Court. 
 
    This program is the implementation of the case study specified in 
    "Software Engineering: an Object-Oriented Perspective," Wiley 2001,   
    by Eric J. Braude.

    The program is free software; you can redistribute it and/or modify it 
    under the terms of the GNU General Public License as published by the 
    Free Software Foundation; either version 2 of the License, or any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    For the GNU General Public License, see http://www.gnu.org/copyleft/gpl.txt 
    or write to the Free Software Foundation, Inc., 
    59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Eric Braude can be reached at ebraude@bu.edu and: Boston University,       
    MET Computer Science Department, 808 Commonwealth Avenue, Boston, MA 02215. 
    Tom Van Court can be reached at tvancour@hotmail.com
*/

import java.awt.*;
import java.io.*;

import Encounter.EncounterCharacters.*;
import Encounter.EncounterEnvironment.*;
import Encounter.EncounterGame.EncounterGameDisplays.*;
import FrameworkRPG.Debug;
import FrameworkRPG.RolePlayingGame.*;

/** Represents the state of an Encounter game in which an engagement
* is currently taking place between the player's character and the foreign
* character. 
* <p> Requirements: SRS 2.1.1 Concept of operations,
*       SRS 3.2.1.3 engagement use case,
*       SRS 3.2.AR.1.3 area-specific qualities,
*       SRS 3.2.AR.4.2 handling engagements,
*       SRS 3.2.AR.4.3 interrupting engagements,
*       SRS 3.2.EN Engagements
* <p> Design: SDD 3.4 State Model Decomposition.
* <br> SDD 6.1.1 Detailed design of <i>RolePlayingGame</i> package
*
* <p> Design issues: <ul>
* <li> One major feature goes way beyond anything stated clearly in the 
*   requirements or design spec. Much is written about how characters find
*   each other in an area, exchange quality points, and so on. All that 
*   seems to happen as an instantaneous event. At very least, that seems
*   likely to surprise a player who might enter an area, engage another
*   character, and be bounced out before ever seeing the area or the other
*   player. It also causes problems with requirement 3.2.AR.4.3, which 
*   specifies how a player may abort an engagement via the user interface.
*   If the engagement is instantaneous, there is no time interval in 
*   which the player can carry out the procedure stated in that requirement.
*   <br> This implementation takes the liberty of adding a time delay
*   to the start of an engagement, and adding a message announcing that
*   that the engagement is beginning. The message appears throughout the
*   idle period before the engagement actually takes place, and represents
*   the period in which the user may abort the engagement.
* </ul>
* @author   Dr. Eric Braude
* @version  0.1
*/

public class Engaging extends EncounterGameState 
{
    /** Character involved in the engagement */
    private EncounterCharacter engagerI;
    
    /** Dialog that announces start of engagement. */
    private Dialog startMessageI;
    
    /** Amount of time we delay before quality exchange */
    private static final int PRE_FIGHT_MILLISEC = 2500;
    
    /** Decide whether the engagement proceeds or aborts. */
    private volatile boolean exitEngagementI = false;   //NTS-1 (end of file)
    
    /** There's some probablity that the user can escape this
    * engagement. The requirement states that there's a certain
    * probability the user can escape the engagement, not some
    * chance of escape on each attempt. Thus, the chance of escape
    * is a property of the engagement as a whole.
    * <p> Requirement: SRS 3.2.AR.4.3 Interrupting engagements,
    */
    private boolean escapableI = Math.random() < 0.1;

    /** Next state to enter after the engagement. */
    private EncounterGameState nextGameStateI;
        
    /*----------------------------------------------------------------------*\
    ** Constructors
    \*----------------------------------------------------------------------*/

    /** Create a new state handler, waiting for game events to occur.
    * @param    gameP       Game context in which player waits.
    * @param    engagerP    Player initiating the engagement
    */
    public Engaging ( EncounterGame gameP,  
        EncounterCharacter engagerP )
    {
        super( gameP );                         // Standard state setup.

        engagerI = engagerP;                    // Initiated engagement.
        
        nextGameStateI = new Waiting( gameP );  // Assume idle state is next.
        
        startMessageI = new StartEngagement();  // Show the start message.
    }
        
    /*----------------------------------------------------------------------*\
    ** Encounter.EncounterGame.EncounterGameState interface
    \*----------------------------------------------------------------------*/

    /** Handle the event in which a character enters a new area.
    * Use the next game state already requested, unless there's a
    * another character in the area. In that case, force the game
    * into an engagement state. 
    *
    * @param    characterP      Character that's moving.
    * @param    areaP           Area into which character moves.
    * @param    encounteringP   <tt>true</tt> iff more than character is 
    *                           now in the area.
    */
    public void handleEvent( EncounterCharacter characterP, 
        Area areaP, boolean encounteringP ) 
    {
        if ( encounteringP )
            nextGameStateI = new Engaging( gameI, characterP );

        gameI.setState( nextGameStateI );
    }

    /*----------------------------------------------------------------------*\
    ** FrameworkRPG.GameEnvironment.GameState interface
    \*----------------------------------------------------------------------*/

    /** Perform any actions that need to happen upon entry into the state.
    * For an engagement, we wait out the initial time delay. Unless the
    * user aborted the engagement, we go ahead and carry it out. Either way,
    * we finish by launching the player into a new area. 
    */
    public void entryActions()
    { 
        super.entryActions();       // Handle superclass default actions. 
 
        // Delay for a while. Watch for user to request 
        // a status report while delaying. 
        long endTimeM = System.currentTimeMillis() + PRE_FIGHT_MILLISEC;        
        while (endTimeM > System.currentTimeMillis() && !exitEngagementI) {
            // Sleep for a little while. Interruptions mean termination.
            // The "sleep" lets other threads run while we wait.
            try { 
                Thread.sleep( 5 );
            } catch (InterruptedException intex) { 
                exitEngagementI = true;
            }
        }
        
        startMessageI.dispose();        // Shut down the start-of-
        startMessageI = null;           // engagement message.

        // If we're not just running away, run the engagement and 
        // display the player character's quality values afterwards.
        if ( !exitEngagementI ) {
            // Run the exchange of quality values.
            // Note which player initiated the engagement.
            ( new Engagement( engagerI, gameI ) ).execute();
            
            // Prepare to enter the state that reports quality values.
            nextGameStateI = new ReportingEncounter( gameI );
        }
        
        // Build list of all areas, except the current one.
        Area engAreaM = gameI.getPlayerArea( engagerI ),
            targetAreasM[] = gameI.getEnvironment().allAreas();
        
        // Remove the current area from the list. 
        int nAreasM = targetAreasM.length;
        for ( int i = 0; i < nAreasM; i++ )
            if ( targetAreasM[i] == engAreaM ) {
                targetAreasM[i] = targetAreasM[nAreasM-1];
                nAreasM -= 1;
                break;
            }
            
        // Move the player's character to a random, new area.
        int randAreaM = (int) Math.floor(Math.random() * nAreasM);
        gameI.movePlayerTo( targetAreasM[randAreaM] );
    }
        
    /** Handle the user interface event that occurs on a request to 
    * change the player's qualities. 
    * <p> Requirements:     SRS 3.2.AR.4.3
    */
    public void setQualities()
    {
        // If we can leave the game, set a flag saying so
        // and note that the user wants to update qualities.
        if ( escapableI ) {
            nextGameStateI = new SettingQualities( gameI );
            exitEngagementI = true;
        }
    }
}

/* NTS - Notes To Students 
* 1: 'volatile' declaration of exitEngagementI. This prevents the variable's
*   value from being cached by the Java optimizer. Supposed Java cached the 
*   value in entryActions(), and used the cached copy as the control value
*   for the 'while' loop. If that happened, then exitEngagementI could not be 
*   set by another thread to change the character state. No matter what the 
*   second thread did, the loop would continue using the stale cached value.
*   A 'volatile' variable is uncacheable; its value has to be read from the 
*   original every time. That means every iteration of the loop checks the 
*   actual value of exitEngagementI, and detects state change requests made by 
*   other threads. 
* 
*/
