package Encounter.EncounterGame;

/* Class Name           : Engagement
* Version information   : Version 0.1
* Date                  : 07/02/1999
* Copyright Notice      : see below
* Edit history:
*   21 Jun 2000 Tom VanCourt    Work on deadlock avoidance. Moved
*                                "synchronized" from method declaration to
*                                a guard block, to avoid recursion while
*                                holding the sync lock.
*   20 May 2000 Tom VanCourt    Treat transfer of points as a game event
*   13 May 2000 Tom VanCourt    Implemented SRS 3.2.EN
*   08 Jan 2000 Tom VanCourt    Change to meet coding standards.
*/

/*
    Copyright (C) 2000 Eric J. Braude and Thomas D. Van Court. 
 
    This program is the implementation of the case study specified in 
    "Software Engineering: an Object-Oriented Perspective," Wiley 2001,   
    by Eric J. Braude.

    The program is free software; you can redistribute it and/or modify it 
    under the terms of the GNU General Public License as published by the 
    Free Software Foundation; either version 2 of the License, or any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    For the GNU General Public License, see http://www.gnu.org/copyleft/gpl.txt 
    or write to the Free Software Foundation, Inc., 
    59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Eric Braude can be reached at ebraude@bu.edu and: Boston University,       
    MET Computer Science Department, 808 Commonwealth Avenue, Boston, MA 02215. 
    Tom Van Court can be reached at tvancour@hotmail.com
*/

import java.awt.*;

import Encounter.EncounterCharacters.*;
import Encounter.EncounterEnvironment.*;
import Encounter.EncounterGame.*;
import FrameworkRPG.Debug;
import FrameworkRPG.RolePlayingGame.*;

/** The Engagement class. Character confront each other, and the
* winner takes some points from the loser.
* <p> Strategy note: A player becomes invincible if all the player's
*   initial quality points are piled into one quality and all players
*   start with the same number of quality points. That player can only
*   tie or beat other players when its one quality value is important
*   in the engagement area, so the player can't lose points. The player
*   necessarily loses to all other players in areas where the player's
*   one quality value is not important. In that case, the player sacrifices
*   50% of all the area-specific quality values, which are all 0. The
*   player loses, but loses nothing.
* <p>Requirements: SRS 3.2.1.3 <i>Encounter</i> use case.
* <br> SRS 3.2.EN
* <br> SRS 3.5.4.3 Altering rules of engagement (can be supported by
*   creating subclasses of this class type)
* <p> Design issues: <ul>
* <li> There's a horrible concurrency exposure in <tt>execute()</tt>.
*   Other processes can read and update character quality values while
*   <tt>execute()</tt> runs around looking at all the characters in the area,
*   then some time later updates the character quality values. Declaring
*   <tt>execute()</tt> as <tt>synchronized</tt wouldn't do any good, because
*   the concurrency might be driven by another object. Using a synchronize
*   block on some character object can lead to deadlock if other updaters
*   take mutex locks on characters in a different order. <br>
*   At this writing, there's no chance for other update operations to
*   run concurrently, so it's not yet a problem. Future enhancement
*   programmers should take care, though.
* </ul>
*
* @author   Dr. Eric Braude
* @version  0.1
*/

public class Engagement
{
    /** Fraction of points forfeited by loser */
    private static final float FORFEITURE = 0.5f;

    /** Identify the character that initiated the engagement. */
    protected EncounterCharacter engagerI;

    /** Identify the game in which the engagement occurs. */
    protected EncounterGame gameI;

    /*----------------------------------------------------------------------*\
    ** Constructors
    \*----------------------------------------------------------------------*/

    /** Default Constructor
    * @param    engagerP    Character whose action initiate the engagement
    * @param    gameP       Game in which engagement occurs
    */
    public Engagement( EncounterCharacter engagerP,
        EncounterGame gameP )
    {
        engagerI = engagerP;
        gameI = gameP;
    }

    /*----------------------------------------------------------------------*\
    ** Own interface
    \*----------------------------------------------------------------------*/

    /** Executes the engagement.
    */
    public void execute()
    {
        Area engagementAreaM = EncounterGame.       // Find the area that
            getTheEncounterGame().                  // the player entered
            getPlayerArea( engagerI );              // to start engaging.

        EncounterCharacter[] areaCharactersM =      // List of characters
            engagementAreaM.getAllCharacters();     // in the area.

        int nCharsM = areaCharactersM.length;       // Number of characters

        if (nCharsM < 2)                            // Make sure that someone
            return;                                 // encounters someone else.

        String[] areaQualitiesM =                   // List of qualities
            engagementAreaM.getAreaQualities();     // that apply in area.

        if ( areaQualitiesM.length == 0 )           // Default is all qualities.
            areaQualitiesM = EncounterCharacter.getQualityTypes();

        float[] charQualTotalM = new                // Quality total for
            float[areaCharactersM.length];          // each player.

        for (int ch = 0; ch < nCharsM; ch++) {      // For each character
            // Save the before-encounter values.
            areaCharactersM[ch].saveQualityValues();

            // Total the area-dependent quality values.
            for (int qp = 0; qp < areaQualitiesM.length; qp++)
                charQualTotalM[ch] += areaCharactersM[ch].
                    getQualityValue( areaQualitiesM[qp] );
        }

        int qualMaxM = 0;                           // Index of highest total
        for (int ch = 1; ch < nCharsM; ch++)        // Find highest total.
            if ( charQualTotalM[ch] > charQualTotalM[qualMaxM] )
                qualMaxM = ch;

        // Find players tied with winner. Note that winner ties itself,
        // so winner does not exchange points with itself.
        boolean tieM[] = new boolean[nCharsM];
        for ( int ch = 0; ch < nCharsM; ch++ )
            tieM[ch] = ( charQualTotalM[ch] == charQualTotalM[qualMaxM] );

        // Winner takes half of other players' points, unless tied.
        for ( int ch = 0; ch < nCharsM; ch++ )
            if ( !tieM[ch] )
                for ( int qp = 0; qp < areaQualitiesM.length; qp++ ) {
                    float winnerPts = areaCharactersM[qualMaxM].
                            getQualityValue( areaQualitiesM[qp] ),
                        loserPts = areaCharactersM[ch].
                            getQualityValue( areaQualitiesM[qp] );

                    // Winner takes points from loser.
                    areaCharactersM[qualMaxM].setQuality( areaQualitiesM[qp],
                            winnerPts + loserPts * FORFEITURE );

                    // Loser loses points.
                    areaCharactersM[ch].setQuality( areaQualitiesM[qp],
                            loserPts * (1.0f-FORFEITURE) );
                }

        // Transfer of points is an event that may affect game state.
        for ( int ch = 0; ch < nCharsM; ch++ )
            gameI.getEncounterState().handleEvent( areaCharactersM[ch] );
    }

    /*----------------------------------------------------------------------*\
    ** Object interface
    \*----------------------------------------------------------------------*/

    /** Generate a printable string summarizing the object.
    * Java defines this method as the type-coercion method for
    * converting this object to a String.
    *
    * @return       Printable string representing this object.
    */
    public String toString()
        { return getClass().getName(); }
}
