package Encounter.EncounterGame.EncounterGameDisplays;

/* Class Name           : SetQualityDisplay
* Version information   : Version 0.1
* Date                  : 04/11/2000
* Copyright Notice      : see below
* Edit history:
*   21 Jun 2000 Tom VanCourt    Added note to student for 'volatile'
*    2 May 2000 Tom VanCourt    Added support for requirement 3.2.PQ.1.1
*                               (color scheme), 3.2.PQ.4.4 (backing out
*                               of a set of changes), and 3.2.PQ.4.3 (time
*                               delay when exiting dialog).
*   16 Apr 2000 Tom VanCourt    Added error logging if asked to display
*                               when already displayed.
*   11 Apr 2000 Tom VanCourt    Initial coding
*/

/*
    Copyright (C) 2000 Eric J. Braude and Thomas D. Van Court. 
 
    This program is the implementation of the case study specified in 
    "Software Engineering: an Object-Oriented Perspective," Wiley 2001,   
    by Eric J. Braude.

    The program is free software; you can redistribute it and/or modify it 
    under the terms of the GNU General Public License as published by the 
    Free Software Foundation; either version 2 of the License, or any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    For the GNU General Public License, see http://www.gnu.org/copyleft/gpl.txt 
    or write to the Free Software Foundation, Inc., 
    59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Eric Braude can be reached at ebraude@bu.edu and: Boston University,       
    MET Computer Science Department, 808 Commonwealth Avenue, Boston, MA 02215. 
    Tom Van Court can be reached at tvancour@hotmail.com
*/

import java.awt.*;
import java.awt.event.*;

import Encounter.EncounterCharacters.PlayerCharacter;
import Encounter.EncounterGame.EncounterGame;
import Encounter.EncounterGame.EncounterGameState;
import FrameworkRPG.Debug;

/** Handle update of player's quality values
* <p> Requirements: SRS 2.1.2.2 User Interface Concept for setting 
*   quality value, 
* <br> SRS 3.2.PQ
* <p> Design issues: <ul>
* <li> This just presents the data stated in the requirements
*   document. It really needs a round of cosmetic improvements.
* <li> This class is derived conceptually from StatusDisplay, but
*   doesn't share as much source as it might. Consider this class
*   a candidate for refactoring.
* <li> This presents the character's name in the window border,
*   rather than the window content as shown in the spec.
* <li> This does not limit the number of digits of precision
*   in the display of quality points.
* <li> Bug: This code explicitly assigns newQualValueI a blue
*   background, as required by SRS. Even so, newQualValueI appears
*   with a white background. Need to do debug work to figure out
*   why the color displayed is not the color assigned by the code.
* <li> After the user hits the "ok" button to update the quality
*   values, two things can happen. A) a timer may elapse and let the
*   changes be applied to the quality values, or B) the foreign
*   character may enter the area and prevent the changes from being
*   applied. Nothing tells the user very clearly which happened.
* <li> This class does not aggregate all the helper classes named in
*   SDD 6.1.1.1. Anonymous classes, etc, made the aggregate much
*   simpler than shown in that design.
* </ul>
* @author   Tom VanCourt
* @version  0.1
*/

public class SetQualityDisplay extends EncounterDisplay
    implements ActionListener, ItemListener, TextListener
{
    private static final String BLANKS =
        "                                                             ";

    /** Info dialog, if it's already shown, null if not shown. */
    private static SetQualityDisplay theQualityDisplayS = null;

    /** Character being updated. */
    private PlayerCharacter characterI = null;

    /** Display status messages about user input.
    * Reserve plenty of space in the message area
    */
    private Label statusMsgI = new Label( BLANKS );

    /** Display the current life point total. */
    private Label lifeTotalI = new Label();

    /** List of character quality names. */
    private List qualListI = new QualValueDisplay();

    /** Text entry for new quality value */
    private TextField newQualValueI = new TextField(10);

    /** "Light turquoise" from 3.2.PQ.1 */
    private static final Color BLUE_DATA = new Color(200, 240, 255);

    /** Delay before closing window and commiting changes to quality values
    * <p> Requirement: 3.2.PQ.4.3
    */
    private static final int CLOSE_DELAY_MS = 4000;     // millisec

    /** When true, this says the dialog should close immediately and prevent 
    * quality value updates from taking effect. False means that the dialog
    * times out and closes normally, updating quality values.
    * This must be volatile, since it may be updated asynchronously to
    * pass a halt request from one thread to another.
    * <p> Requirement: 3.2.PQ.4.4
    */
    private volatile boolean updateAbortI = false;      //NTS-1 (end of file)

    /*----------------------------------------------------------------------*\
    ** Constructor
    \*----------------------------------------------------------------------*/

    private SetQualityDisplay( PlayerCharacter characterP,
        EncounterGameState displayingStateP )
    {
        super(                                  // Create the dialog.
            EncounterGame.getTheEncounterGame().getMainFrame(),
            characterP.getName() + " quality update",   // Dialog name
            false,                              // Do not show until set up.
            displayingStateP);                  // State that wants display.

        characterI = characterP;                // Remember the character.
        Panel topRowM = new Panel(),            // Panels for organizing UI
            midRowM = new Panel(),              //  row by row.
            botRowM = new Panel();

        add( topRowM, BorderLayout.NORTH );     // Display layout
        add( midRowM, BorderLayout.CENTER );
        add( botRowM, BorderLayout.SOUTH );

        topRowM.add(                            // Character image
            new CharacterCanvas( characterP ) );

        topRowM.add( lifeTotalI );              // Life point total.
        showTotal();

        qualListI = new QualValueDisplay();
        qualListI.addItemListener( this );      // Respond to list events.
        midRowM.add( qualListI );               // Display quality values

        midRowM.add( newQualValueI );           // Text entry area
        newQualValueI.addTextListener( this );  // Respond to typing.

        qualListI.select( 0 );                  // Select first in the list.
        itemStateChanged( null );               // Update the caption.

        botRowM.add( statusMsgI );              // Status message at bottom.

        Button doneM = new Button( "OK" );      // Button to close status.
        doneM.addActionListener( this );        // Destroy status dialog.
        botRowM.add( doneM );                   // Add the "done" button.

        Component whiteUI[] = { topRowM,        // Use white background.
                midRowM, botRowM },
            blueUI[] = { newQualValueI,         // Colored background.
                lifeTotalI };

        for ( int i = 0; i < whiteUI.length; i++ )      // Apply colors
            whiteUI[i].setBackground( Color.white );
        for ( int i = 0; i < blueUI.length; i++ )
            blueUI[i].setBackground( BLUE_DATA );

        pack();                                 // Pack dialog contents
    }

    /*----------------------------------------------------------------------*\
    ** Encounter.EncounterGame.EncounterGameDisplays.EncounterDisplay
    \*----------------------------------------------------------------------*/

    /** Force the display to shut down immediately. */
    public void forceQuit()
    {
        updateAbortI = true;            // Don't let update take effect.
        actionPerformed(null);          // Only supported action is 'quit'.
    }

    /*----------------------------------------------------------------------*\
    ** Interface defined by this class
    \*----------------------------------------------------------------------*/

    /** Adjust quality values for a specified character.
    * @param    characterP      Character for which to adjust qualities.
    *                           Must be non-null.
    * @return                   Display (dialog) object
    */
    public static EncounterDisplay showDisplay( PlayerCharacter characterP,
        EncounterGameState displayingStateP )
    {
        // If some other character is being shown, take down that display.
        if (theQualityDisplayS != null)
            if (characterP != theQualityDisplayS.characterI) {
                Debug.log("SetQualityDisplay requested when showing");
                theQualityDisplayS.forceQuit( );
            }

        if (theQualityDisplayS == null) {           // Create if needed.
            theQualityDisplayS = new SetQualityDisplay( characterP,
                displayingStateP );

            characterP.updateBegin();               // Start of update.

            // Set up handler for accepting new input.
            theQualityDisplayS.newQualValueI.addActionListener(
                new ActionListener() {
                    public void actionPerformed( ActionEvent actP )
                        { theQualityDisplayS.takeInputValue(); }
                } );

            theQualityDisplayS.addWindowListener(   // Normal shutdown.
                new WindowAdapter() {
                    public void windowClosing( WindowEvent e )
                        { theQualityDisplayS.actionPerformed( null ); }
                } );
        };

        theQualityDisplayS.toFront();               // Jam it in user's face.
        theQualityDisplayS.show();                  // Make sure it shows.

        return theQualityDisplayS;                  // Send dialog to caller
    }

    /** Update the display of life points.
    * This method has such a simple pattern of data reference that
    * it doesn't need to be synchronized.
    */
    private void showTotal()
    {
        lifeTotalI.setText( "Total life points: " +
            characterI.sumOfQualities() );
    }

    /** Accept a new quality value from input.
    */
    private void takeInputValue()
    {
        statusMsgI.setText( BLANKS );               // Clear old status.

        String inputM = newQualValueI.getText();    // Accept new string.

        try {
            float inputVal =                        // Convert input to number.
                Float.valueOf( inputM ).floatValue();

            // Report input error or accept new quality value.
            if (inputVal < 0)
                // Not big enough
                statusMsgI.setText( "Input must be at least zero." );
            else if (inputVal > characterI.sumOfQualities() )
                // Too big
                statusMsgI.setText( "Input must not exceed " +
                    characterI.sumOfQualities() );
            else {
                // Value was in legal range.
                int index = qualListI.getSelectedIndex( );
                String quality = qualListI.getItem( index );
                characterI.adjustQuality( quality, inputVal );
                statusMsgI.setText( "New value accepted: " + inputVal );
            }

        } catch (NumberFormatException nfex) {      // Show conversion message.
            statusMsgI.setText( "Please enter a number 0-" +
                characterI.sumOfQualities() );
        }

        // Setting a quality value can change the total.
        // Make sure the total stays up to date.
        showTotal();
    }

    /*----------------------------------------------------------------------*\
    ** ActionListener interface
    \*----------------------------------------------------------------------*/

    /** This dialog may shut down because of user activity, because an
    * encounter demands that it shut down, or (unfortunately) both at once.
    * This flag synchronizes multiple shutdown requests with each other.
    */
    private boolean shuttingDownI = false;              // not shutting down

    /** Handle input from the shutdown button: clean shutdown.
    * Delay for a specified amount of time, then commit the updates and
    * close the window. Rather than hang silently during the shutdown
    * delay, blink a message to show something's running.
    * <p> This is NOT synchronized, since a foreign character may enter
    * the area during the shutdown interval and want to prevent quality
    * value updates from going into effect.
    * <p> It's possible for two or more shutdown requests to meet in this
    * method. The first thread in completes the shutdown. All other threads
    * see the first one and assume it will finish the job for them. Callers
    * can not assume that the dialog is gone and shutdown complete when
    * this method returns.
    *
    * Precondition:         Dialog must exist.
    * Postcondition:        Dialog has been deleted.
    * @param    actionP     Input event -- can only be 'done' button
    *                       or equivalent shutdown request.
    */
    public void actionPerformed( ActionEvent actionP )
    {
        // Note: Time units in this method are millisec (abbr. ms, Ms, MS).

        final int BLINK_MS = 333;               // Blink interval, millisec.

        // Very short mutex region to test&set shutdown flag.
        synchronized(this) {                    // Avoid concurrency problems.
            if ( shuttingDownI )                // is someone already closing?
                return;                         // If so, just let it run.
            else                                // If not, we claim the
                shuttingDownI = true;           //  flag and do the job.
        }

        boolean blinkOnM = true;                // On/off blinker.
        long
            // Time at which shutdown finishes.
            endClosingMsM = System.currentTimeMillis() + CLOSE_DELAY_MS,
            // Time at which next blink occurs.
            blinkMsM = System.currentTimeMillis() + BLINK_MS;

        // Keep going until it's really shutdown time, or we're told to quit.
        // Poll the wall clock with 1ms pauses between samples.
        while (!updateAbortI && System.currentTimeMillis() < endClosingMsM) {
            if (System.currentTimeMillis() > blinkMsM) {
                // Blink the shutdown message and pick a future blink time.
                statusMsgI.setText( (blinkOnM ^= true) ?
                    "Saving ..." : BLANKS );
                blinkMsM = System.currentTimeMillis() + BLINK_MS;
            }

            // Sleep for a little while. Interruptions mean termination.
            // The "sleep" lets other thread run while we wait.
            try {
                Thread.sleep( 5 );
            } catch (InterruptedException intex) {
                updateAbortI = true;
            }
        }

        // The update period is done. Roll forward or roll back, depending
        // on whether the update was aborted.
        characterI.updateEnd(!updateAbortI);

        theQualityDisplayS.dispose();           // Destroy the dialog object
        theQualityDisplayS = null;

        displayingStateI.handleEvent( actionP );    // State-dependent action
    }

    /*----------------------------------------------------------------------*\
    ** ItemListener interface
    \*----------------------------------------------------------------------*/

    /** Handle input from the list that selects a quality to display.
    * @param    actionP     Input event -- can only be quality list selection
    */
    public void itemStateChanged( ItemEvent e )
    {
        float itemValueM;
        
        // New user action invalidates any older response message.
        statusMsgI.setText( BLANKS );

        // Load input area with old quality value.
        synchronized( qualListI ) {
            int indexM = qualListI.getSelectedIndex( );
            String itemM = qualListI.getItem( indexM );
            itemValueM = characterI.getQualityValue( itemM );
        }
        newQualValueI.setText( itemValueM + "" );
    }

    /*----------------------------------------------------------------------*\
    ** TextListener interface
    \*----------------------------------------------------------------------*/

    /** Handle input that says a user is typing.
    * @param    textP       Input event -- results from input edits
    */
    public void textValueChanged( TextEvent textP )
    {
        // New user action invalidates any older response message.
        statusMsgI.setText( BLANKS );
    }
}

/* NTS - Notes To Students 
* 1: 'volatile' declaration of updateAbortI. This prevents the variable's
*   value from being cached by the Java optimizer. Supposed Java cached the 
*   value in actionPerformed(), and used the cached copy as the control value
*   for the 'while' loop. If that happened, then updateAbortI could not be 
*   set by another thread to terminate the loop early. No matter what the 
*   second thread did, the loop would continue using the stale cached value.
*   A 'volatile' variable is uncacheable; its value has to be read from the 
*   original every time. That means every iteration of the loop checks the 
*   actual value of updateAbortI, and detects changes made by other threads. 
*/
