package Encounter.EncounterGame.EncounterGameDisplays;

/* Class Name           : EncounterDisplay
* Version information   : Version 0.1
* Date                  : 04/10/2000
* Copyright Notice      : see below
* Edit history:
*   21 Jun 2000 Tom VanCourt    Work on deadlock avoidance. Moved 
*                                "synchronized" from method declaration to 
*                                a guard block, to avoid recursion while 
*                                holding the sync lock.
*   13 May 2000 Tom VanCourt    Added forceQuit
*    2 May 2000 Tom VanCourt    Added support for requirement 3.2.AR.4.6
*                               (values before last encounter)
*   16 Apr 2000 Tom VanCourt    Added error logging if asked to display
*                               when already displayed.
*   10 Apr 2000 Tom VanCourt    initial coding
*/

/*
    Copyright (C) 2000 Eric J. Braude and Thomas D. Van Court. 
 
    This program is the implementation of the case study specified in 
    "Software Engineering: an Object-Oriented Perspective," Wiley 2001,   
    by Eric J. Braude.

    The program is free software; you can redistribute it and/or modify it 
    under the terms of the GNU General Public License as published by the 
    Free Software Foundation; either version 2 of the License, or any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    For the GNU General Public License, see http://www.gnu.org/copyleft/gpl.txt 
    or write to the Free Software Foundation, Inc., 
    59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Eric Braude can be reached at ebraude@bu.edu and: Boston University,       
    MET Computer Science Department, 808 Commonwealth Avenue, Boston, MA 02215. 
    Tom Van Court can be reached at tvancour@hotmail.com
*/

import java.awt.*;
import java.awt.event.*;

import Encounter.EncounterCharacters.PlayerCharacter;
import Encounter.EncounterGame.EncounterGame;
import Encounter.EncounterGame.EncounterGameState;
import FrameworkRPG.Debug;

/** Handle display of the character status
* <p> Requirement:  SRS 3.2.ED, 
*                   SRS 3.2.1.3 <i>Encounter</i> use case.
* <p> Design issues: <ul>
* <li> This just presents the data stated in the requirements
*   document. It really needs a round of cosmetic improvements.
* <li> Quality points are not displayed in the proper format,
*   with exactly one digit right of the decimal.
* <li> This presents the character's name in the window border,
*   rather than the window content as shown in the spec.
* <li> This class does not aggregate all the helper classes named in
*   SDD 6.1.1.1. Anonymous classes, etc, made the aggregate much
*   simpler than shown in that design.
* <li> This class differs from SRS 3.4.AR.4.6: it does not display the
*   player's quality values as they were after the most recent
*   encounter. Instead, this displays the player's current quality
*   values including any player-directed changes since the most recent
*   encounter. I couldn't imagine any reasonable circumstance in which
*   a player would want the values after the encounter rather than the
*   current values, so took the liberty of over-riding the spec.
* <li> SDD 6.1.1.2.2 uses "setVisible" to dismiss a dialog box, but
*   this in fact uses actionPerformed (a button handler) and 
*   java.awt.Dialog.dispose()
* </ul>
* @author   Tom VanCourt
* @version  0.1
*/

public class EngagementDisplay extends EncounterDisplay
    implements ActionListener, ItemListener
{
    /** Info dialog, if it's already shown, null if not shown. */
    protected static EngagementDisplay statusDisplayS = null;

    /** List of character quality names. */
    protected List qualListI = new QualValueDisplay();

    /** Place to report quality values. */
    protected Label qualValueI = new Label( "     " );

    /** Place to report previous quality values. */
    protected Label oldValueI = new Label( "     " );

    /** Character being described. */
    protected PlayerCharacter characterI = null;
    
    /*----------------------------------------------------------------------*\
    ** Constructor
    \*----------------------------------------------------------------------*/

    private EngagementDisplay( PlayerCharacter characterP,
        EncounterGameState displayingStateP )
    {
        super(                                  // Create the dialog.
            EncounterGame.getTheEncounterGame().getMainFrame(),
            characterP.getName() + " status",   // Dialog name
            false,                              // Do not show until set up.
            displayingStateP);                  // State that wants display.

        characterI = characterP;                // Remember the character.

        Panel topRowM = new Panel(),            // Panels for organizing UI
            valuesM = new Panel(),
            botRowM = new Panel();              //  row by row.

        add( topRowM, BorderLayout.NORTH );     // Display layout
        add( botRowM, BorderLayout.SOUTH );

        topRowM.add(                            // Character image
            new CharacterCanvas( characterP ) );
        topRowM.add(                            // Status point label
            new Label("Current life points: " +
                characterP.sumOfQualities() ) );

        qualListI.addItemListener( this );      // Respond to list events.
        botRowM.add( qualListI );               // Display quality values

        Component valTableM[] = {               // Items in value display
            new Label("Current value:"),
            qualValueI,
            new Label(" "),
            new Label("Before last encounter:"),
            oldValueI};

        valuesM.setLayout(                      // Set up the value display.
            new GridLayout(valTableM.length, 1));
        for (int i = 0; i < valTableM.length; i++)
            valuesM.add(valTableM[i]);
        botRowM.add( valuesM );

        qualListI.select( 0 );                  // Select first in the list.
        itemStateChanged( null );               // Update the caption.

        Button doneM = new Button( "Done" );    // Button to close status.
        doneM.addActionListener( this );        // Destroy status dialog.
        botRowM.add( doneM );                   // Add the "done" button.

        Component whiteUI[] = { topRowM,        // Use white background.
                botRowM };

        for ( int i = 0; i < whiteUI.length; i++ )
            whiteUI[i].setBackground( Color.white );

        pack();                                 // Pack dialog contents.
    }

    /*----------------------------------------------------------------------*\
    ** Interface defined by this class
    \*----------------------------------------------------------------------*/

    /** Display status for a specified character.
    * @param    characterP      Character for which to display status.
    * @param    displayingStateP    Game state that requested the display.
    * @return                   The dialog object on screen
    */
    public static EncounterDisplay showDisplay( PlayerCharacter characterP,
        EncounterGameState displayingStateP )
    {
        // If some other character is being shown, take down that display.
        System.out.println("+showDisplay");

        if (statusDisplayS != null)
            if (characterP != statusDisplayS.characterI) {
                Debug.log("EngagementDisplay requested when showing");
                statusDisplayS.actionPerformed( null );
            }

        if (statusDisplayS == null) {               // Create if needed.
            statusDisplayS = new EngagementDisplay( characterP,
                displayingStateP );

            statusDisplayS.addWindowListener(       // Normal shutdown.
                new WindowAdapter() {
                    public void windowClosing( WindowEvent e )
                        { statusDisplayS.actionPerformed( null ); }
                }
            );
        };

        statusDisplayS.toFront();                   // Jam it in user's face.
        statusDisplayS.show();                      // Make sure it shows.
        
        System.out.println("-showDisplay");

        return statusDisplayS;
    }

    /*----------------------------------------------------------------------*\
    ** Encounter.EncounterGame.EncounterGameDisplays.EncounterDisplay
    \*----------------------------------------------------------------------*/

    /** Force the display to shut down immediately. */
    public void forceQuit()
    {
        if (statusDisplayS != null)     // If display is active, shut it down.
            actionPerformed(null);      // Only supported action is 'quit'.
            
        super.forceQuit();              // Superclass shudown, too. 
    }

    /*----------------------------------------------------------------------*\
    ** ActionListener interface
    \*----------------------------------------------------------------------*/

    /** Handle input from the shutdown button.
    * Always return the game to the Waiting state.
    * <p> Design:   SDD 3.4, SDD 6.1.1.2.2
    * @param    actionP     Input event -- can only be 'done' button.
    */
    public void actionPerformed( ActionEvent actionP )
    {
        synchronized ( this ) {
            if (statusDisplayS != null) {
                statusDisplayS.dispose();
                statusDisplayS = null;
            }
        }

        displayingStateI.handleEvent( actionP );
    }

    /*----------------------------------------------------------------------*\
    ** ItemListener interface
    \*----------------------------------------------------------------------*/

    /** Handle input from the list that selects a quality to display.
    * @param    actionP     Input event -- can only be quality list selection
    */
    public void itemStateChanged( ItemEvent actionP )
    {
        int indexM;                     // Index of the chosen quality
        String itemM;                   // Name of the chosen quality
        float oldValueM, newValueM;     // Previous and recent quality values
        
        synchronized( qualListI ) {             // Hold quality list stable.
            indexM = qualListI.getSelectedIndex( );
            itemM = qualListI.getItem( indexM );
        }
        
        synchronized( characterI ) {            // Hold character stable
            newValueM = characterI.getQualityValue( itemM );
            oldValueM = characterI.getOldValue( itemM );
        }

        qualValueI.setText( newValueM + "" );   // Update UI
        oldValueI.setText( oldValueM + "" );
    }
}
