package Encounter.EncounterGame;

/* Class Name           : EncounterGame
* Version information   : Version 0.1
* Date                  : 07/02/1999
* Copyright Notice      : see below
* Edit history:
*   20 May 2000     Tom VanCourt    Added getEnvironment, getEncounterState
*   13 May 2000     Tom VanCourt    Extended getPlayerArea with a parameter
*   10 Apr 2000     Tom VanCourt    Added Frame param to setDisplay,
*           needed for displaying dialog boxes.
*    3 Apr 2000     Tom VanCourt    Rework after inspection: fix a 
*           cut & paste bug in showStatus
*   21 Mar 2000     Tom VanCourt    Added redisplay.
*   20 Mar 2000     Tom VanCourt    Added showStatus, setQualities.
*           Started the task that runs the foreign character.
*   13 Mar 2000     Tom VanCourt    Added drawPlayerArea, put player
*           into initial game area, added foreignMovementI,
*           added movePlayerTo
*   10 Mar 2000     Tom VanCourt    Incorporate review comments.
*   08 Mar 2000     Tom VanCourt    Record time at which game starts.
*           Added interfaces from design statement 5.1.1, drawThumbnail, 
*           stateI, gameCastI, gameEnvironI, thumbnailMapI.
*   08 Jan 2000     Tom VanCourt    Change to meet coding standards.
*/

/*
    Copyright (C) 2000 Eric J. Braude and Thomas D. Van Court. 
 
    This program is the implementation of the case study specified in 
    "Software Engineering: an Object-Oriented Perspective," Wiley 2001,   
    by Eric J. Braude.

    The program is free software; you can redistribute it and/or modify it 
    under the terms of the GNU General Public License as published by the 
    Free Software Foundation; either version 2 of the License, or any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    For the GNU General Public License, see http://www.gnu.org/copyleft/gpl.txt 
    or write to the Free Software Foundation, Inc., 
    59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Eric Braude can be reached at ebraude@bu.edu and: Boston University,       
    MET Computer Science Department, 808 Commonwealth Avenue, Boston, MA 02215. 
    Tom Van Court can be reached at tvancour@hotmail.com
*/

import java.awt.*;
import java.awt.AWTEvent.*;
import java.util.Date;

import Encounter.EncounterCharacters.*;
import Encounter.EncounterEnvironment.*;
import FrameworkRPG.Debug;
import FrameworkRPG.RolePlayingGame.*;

/** The EncounterGame class. This collects  the players, the areas 
* where play occurs, etc. into one interface.
* <p> Requirements: SRS 3.2.1.1 <i>Initialize</i> use case,
* <br> SRS 3.2.1.3 <i>Encounter</i> use case.
* <br> SRS 3.2.EG
* <p> Design: SDD 3.1 Module decomposition
* <br> SDD 5.1.1 Interface to the EncounterGame package
*
* <p> Design issues: <ul>
* <li> This aggregates the ThumbnailMap object, which is not
*   named in the design.
* <li> SDD 5.1.3 says that movePlayerTo() should have been implemented
*   in the EncounterEnvironment package instead of here.
* </ul>
*
* @author   Dr. Eric Braude
* @version  0.1
*/

public class EncounterGame extends RPGame 
{
    /** Singleton instance of the game object. 
    * <p> Requirement SRS 3.2.EG.2.1 Single game
    */
    protected static final EncounterGame theEncounterGameS = 
        new EncounterGame();
    
    /** Keep track of game duration. Object instantiation
    * is taken to represent the time at which the game starts.
    * <p> Requirement 3.2.EG.1.1 
    */
    protected final Date startTimeI = new Date();
    
    /** Characters now active in the game. 
    * <p> Design: SDD 4.1
    */
    protected final EncounterCast gameCastI = 
        EncounterCast.getEncounterCast();
    
    /** Labyrinth environment in which game is played. 
    * <p> Design: SDD 4.1
    */
    protected final EncounterEnvironment gameEnvironI;
        
    /** Thumbnail map that summarizes the local part of the labyrinth. */
    private final ThumbnailMap thumbnailMapI = ThumbnailMap.getMap();
    
    /** Main display area for game. 
    * Initialized to a dummy object that eliminates the need to 
    * check for valid UI objects. 
    */
    private Component mainDisplayI = new Label( "dummy object" );
    
    /** Frame in which game is played, needed for dialog boxes. */
    private Frame mainFrameI = null;
    
    /** Task for handling foreign character. */
    private CharacterMovement foreignMovementI;
     
    /*----------------------------------------------------------------------*\
    ** Constructors
    \*----------------------------------------------------------------------*/

    /** Default Constructor
    */
    private EncounterGame() 
    {
        super();                                // Normal superclass init.
        
        setState( new SettingUp( this ) );      // Safe, idle state for init.
        
        gameEnvironI =                          // Create the game environment object. 
            EncounterEnvironment.getEncounterEnvironment(this) ;
        
        EncounterCharacter playerM =            // Fetch the player.
            gameCastI.getThePlayerCharacter();
        Area firstAreaM =                       // Player's initial area.
            gameEnvironI.getFirstArea();       
        firstAreaM.addCharacter( playerM );     // Set player in first area.
        
        setState( new Waiting(this) );          // First playable state.
        
        foreignMovementI = new CharacterMovement(   // Task for handling
            gameCastI.getTheForeignCharacter(),     // the foreign character
            gameEnvironI, this );                   // in this environment.
        foreignMovementI.start();               // Start the opponent.
    }

    /*----------------------------------------------------------------------*\
    ** Interface defined by this class
    \*----------------------------------------------------------------------*/

    /** Repaint the area containing the player.
    * @param    drawAreaP   UI component in which to draw the area.
    */
    public void drawPlayerArea( Component drawAreaP )
    { 
        //Debug.traceback();
        getPlayerArea().drawArea( drawAreaP );  // Repaint that area. 
    }
    
    /** Repaint the thumbnail map.
    * @param    drawAreaP   UI component in which to draw the map.
    */
    public void drawThumbnail( Component drawAreaP )
        { thumbnailMapI.drawMap( drawAreaP ); }
        
    /** Fetch the current state of the game
    * @return   game's current state object
    */
    public EncounterGameState getEncounterState()
        { return (EncounterGameState) getState(); }
        
    /** Fetch the game's geographic environment.
    * @return   The environment for the current game.
    */
    public EncounterEnvironment getEnvironment()
        { return gameEnvironI; }
        
    /** Fetch the main game-playing frame 
    * @return   Main game-playing frame. 
    */
    public Frame getMainFrame()
        { return mainFrameI; }
        
    /** Fetch the area that contains the player. 
    * @return   Non-null area in which the player resides.
    */
    public Area getPlayerArea()
        { return getPlayerArea( gameCastI.getThePlayerCharacter() ); }
    
    /** Fetch the area that contains some character. 
    * @param    playerP     Character whose area is sought.
    * @return   Non-null area in which the player resides. 
    */
    public Area getPlayerArea( EncounterCharacter playerP )
    {
        Area playerAreaM =                      // Fetch the player's area.
            gameEnvironI.getCharacterArea( playerP );
            
        Debug.assert(playerAreaM != null);
        
        return playerAreaM;
    }
    
    /** Returns the EncounterGame singleton object.
    * <p> Requirement: SDD 5.1.1
    * @return       The singleton object.
    */
    public static EncounterGame getTheEncounterGame() 
        { return theEncounterGameS; }
        
    /** Move the player character to a new area.
    * Implicitly remove it from the area, if any, where it had been.
    * <p>Precondition:      toAreaP is non-null
    * <p>Design statement:  5.1.3
    * @param    toAreaP     Area to which player moves. 
    * @return   <tt>true</tt> if player moves to an area
    *           already containing another player.
    */
    public void movePlayerTo ( Area toAreaP )
    {
        EncounterCharacter playerM =            // Fetch the player.
            gameCastI.getThePlayerCharacter();
            
        Area oldAreaM = getPlayerArea();        // Area where player was.
        
        if (oldAreaM!= null)                    // If player was somewhere,
            oldAreaM.removeCharacter( playerM );//  remove it from there.
            
        toAreaP.addCharacter( playerM ); 
    }
    
    /** If the named area is now on screen, update the display of that area.
    * @param    changedAreaP    Area whose content changed and may need
    *                           to have its on-screen image updated.
    */
    public void redisplay( Area changedAreaP )
    {
        Debug.log("redisplay " + changedAreaP + "/" + getPlayerArea());
        if ( changedAreaP == getPlayerArea() ) {
            Graphics mainGraphics = mainDisplayI.getGraphics();
            mainDisplayI.paint( mainGraphics );
        }
    }
    
    /** Set the display object.
    * @param    mainFrameP      Frame in which game display appears.
    * @param    mainDisplayP    Main UI object for refreshing display.
    */
    public void setDisplay( Frame mainFrameP, Component mainDisplayP )
    { 
        mainDisplayI = mainDisplayP; 
        mainFrameI = mainFrameP;
    }
    
    /** Answer the user's request for a change of player quality values. 
    * Answering the request depends on the current state of the game.
    */
    public void setQualities()
    {
        EncounterGameState myStateM = (EncounterGameState) getState();
        myStateM.setQualities();
    }
    
    /** Answer the user's request for a display of the game status.
    * Answering the request depends on the current state of the game.
    */
    public void showStatus()
    {
        EncounterGameState myStateM = (EncounterGameState) getState();
        myStateM.showStatus();
    }
    
    /*----------------------------------------------------------------------*\
    ** Test interface
    \*----------------------------------------------------------------------*/

    /** Test Code
    *
    * @param        argsP Command line arguments.
    */
    public static void main(String[] argsP) 
    {
        EncounterGame encounterM = getTheEncounterGame();

        /* To test, put in Encountering state and handle event */
        //encounterM.setState(new Encountering());
        //encounterM.handleEvent(new RPGEvent(new Object(), 0)); // dummies

    }  
}
