package Encounter.EncounterGame;

/* Class Name           : CharacterMovement
* Version information   : Version 0.1
* Date                  : 03/01/2000
* Copyright Notice      : see below
* Edit history:
*   21 Jun 2000 Tom VanCourt    Fixed a cold-start bug: don't place
*                   character in the labyrinth until after the 
*                   cold-start delay. That lets the UI thread finish
*                   setting up the display. Added NTSs about volatile
*                   variables and the backstop handler.
*   20 May 2000 Tom VanCourt    Removed logic for changing game state
*                   on character movement.
*   17 May 2000 Tom VanCourt    Made time delays depend on debug switch.
*   12 May 2000 Tom VanCourt    Initiate engagement upon entering an
*                   area containing another character.
*   20 Mar 2000 Tom VanCourt    Added a last-chance (a.k.a. "backstop")
*                   exception handler to run().
*    6 Mar 2000 Tom VanCourt    Incorporated review comments.
*    1 Mar 2000 Tom VanCourt    Initial coding
*/

/*
    Copyright (C) 2000 Eric J. Braude and Thomas D. Van Court. 
 
    This program is the implementation of the case study specified in 
    "Software Engineering: an Object-Oriented Perspective," Wiley 2001,   
    by Eric J. Braude.

    The program is free software; you can redistribute it and/or modify it 
    under the terms of the GNU General Public License as published by the 
    Free Software Foundation; either version 2 of the License, or any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    For the GNU General Public License, see http://www.gnu.org/copyleft/gpl.txt 
    or write to the Free Software Foundation, Inc., 
    59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Eric Braude can be reached at ebraude@bu.edu and: Boston University,       
    MET Computer Science Department, 808 Commonwealth Avenue, Boston, MA 02215. 
    Tom Van Court can be reached at tvancour@hotmail.com
*/

import FrameworkRPG.Debug;
import Encounter.EncounterCharacters.*;
import Encounter.EncounterEnvironment.*;
import Encounter.EncounterGame.*;

/** This class controls a "foreign" player. There is once instance 
* of this class for each foreign player.
* <p> This class implements control via a state machine. The state 
* value is held in nextStateI, and state transitions are computed in
* the transition method. A time delay occurs after each transition, 
* to pace the character to real-world events. 
* <p> Other classes can change the behavior of the state machine. If
* the controlled player needs to respond immediately to some condition, 
* the caller should stuff new state information into nextStateI, then
* interrupt this object. That interrupt kicks the control task out
* of its time delay and into its next state, selected by the new
* nextStateI value.
*
* <p> Requirements: SRS 3.2.FC.3.1 Foreign Character Movement, 
*                   SRS 3.2.1.3 <i>Encounter</i> use case.
* <p> Design:       SDD 3.2 Concurrent process decomposition, 
*                   SDD 5.2.2 Foreign Character Movement Process
*                   SDD paragraph 6.1.1, 
*                   SDD 6.1.1.CM, SDD 5.2.2
* <p> Design issues:<ul>
* <li> This class is not named in the SDD.
* </ul>

* @author   Tom VanCourt
* @version  0.1
*/

public class CharacterMovement extends java.lang.Thread
{
    /** Helps generate thread names. */
    private static final String MOVE = "move-";
    
    /** Limit on random values generated.
    * @see randAvg
    */
    private static final double RAND_LIMIT = 0.1;
    
    // Values for the character control state machine
    
    /** Special state representing creation.
    * Handle any initialization that remains to be done.
    */
    protected static final int STATE_STARTUP = 0;
    
    /** The character jumps to an adjacent area. */
    protected static final int STATE_JUMP_AREA = 1;
    
    /** The character will optimize itslef for the area. */
    protected static final int STATE_OPT_AREA = 2;
    
    /** The character has passed on. Shut down and clean up. */
    protected static final int STATE_DEAD = 3;
    
    /** Millisec delay after optimizing character to the area's 
    * salient qualities.
    * <p> Requirement 3.2.FC.3.1
    */
    private static final int MS_DELAY_AFTER_OPT = 
        Debug.isDebugging() ?           // Slow down for debug.
        3000:                           // Slower debug pace.
        1000;                           // Faster for normal play.
    
    /** Millisec delay after jumping to to a new area.
    * This plus MS_DELAY_AFTER_OPT gives the average time
    * spent in the area. 
    * <p> Requirement 3.2.FC.3.1
    */
    private static final int MS_DELAY_AFTER_JUMP =  
        Debug.isDebugging() ?           // Slow down for debug.
        3000:                           // Slower debug pace.
        1000;
    
    /** Millisec delay after cold start before activating the
    * character. The delay lets other system components finish
    * initializing before the foreign character starts moving around.
    */
    private static final int MS_DELAY_IN_COLDSTART = 5000;
    
    /** Character controlled by this movement object. */
    protected ForeignCharacter characterI;
    
    /** Environment in which character is controlled. */
    protected EncounterEnvironment environmentI;
    
    /** Game in which plyer moves */
    protected EncounterGame gameI;
    
    /** Effectively eternal wait value */
    protected static final long FOREVER = Long.MAX_VALUE;
    
    /** Package a behavioral state with a delay value. 
    * This class is static to allow creation of static instances.
    */
    private static class StateAndDelay {
        /** State defining character's current behavior. */
        int state;
        
        /** Number of millisec of delay after entering state. */
        long postDelayMillisec;
        
        public StateAndDelay( int stateP, long postDelayMillisecP )
        {
            state = stateP;
            postDelayMillisec = postDelayMillisecP;
        }
    }
    
    /** Special startup condition. */
    private static final StateAndDelay coldStartS = 
        new StateAndDelay( STATE_STARTUP, MS_DELAY_IN_COLDSTART );
        
    /** Standard dead condition. */
    protected static final StateAndDelay characterDiedS =
        new StateAndDelay( STATE_DEAD, FOREVER );
    
    /** Character's behavioral state. 
    * Outside callers communicate with the control thread through 
    * asynchronous changes to this value, so the optimizer must not 
    * assume the value remains unchanged from moment to moment.
    * The <tt>volatile</tt> declaration makes this safe for
    * asynchronous changes.
    */
    volatile StateAndDelay nextStateI = coldStartS; // NTS-1 (end of file)
    
    /*----------------------------------------------------------------------*\
    ** Constructors
    \*----------------------------------------------------------------------*/

    /*----------------------------------------------------------------------*\
    ** Constructors
    \*----------------------------------------------------------------------*/

    /** Construct a new control object corresponding to a 
    * foreign character.
    *
    * <p> Design:               SDD 6.1.1.CM
    * @param    characterP      Character controlled by this movement object.
    * @param    environmentP    Environment in which character lives.
    * @param    gameP           Game being played
    */
    public CharacterMovement( ForeignCharacter characterP,
        EncounterEnvironment environmentP,
        EncounterGame gameP)
    {
        characterI = characterP;
        environmentI = environmentP;
        gameI = gameP;
        
        setName( MOVE + characterI.getName() );        
    }
    
    /*----------------------------------------------------------------------*\
    ** java.lang.Thread interface
    \*----------------------------------------------------------------------*/
    
    /** Main execution loop controlling character behavior.
    * This should not be declared <tt>synchronized</tt>. If
    * it were synchronized, then it would block any control
    * methods from executing. 
    * <p> This method acts as the foreign character's execution 
    * environment, so it provides a backstop handler for stray exceptions.
    * Any exception that's a subclass of RuntimeException can be thrown
    * by any method without a "throws" declaration -- that's what the
    * backstop handler is meant for. Stray exceptions should occur only
    * during debug (the handler is a nice place for a debugger breakpoint), 
    * but is harmless in normal execution.
    *
    * <p> Design:               SDD 6.1.1.CM
    */
    public void run()
    {
        try {
            while ( nextStateI.state != STATE_DEAD ) {
                // Remember how long to wait after entering this state.
                long waitMillisec = nextStateI.postDelayMillisec;
                
                // Enter new state and learn next state.
                nextStateI = transition( nextStateI.state );
                
                Debug.log("sleep " + waitMillisec);
                
                try {
                    sleep( waitMillisec );                  // Defined delay.
                } catch( InterruptedException iex ) {
                    // Interruption is the normal way to communicate with the 
                    // character's control thread. Assume nextStateI has been 
                    // changed by some outside caller and that its updated state 
                    // value reflects the new action required.
                }
            }
        } catch ( Throwable thrownException ) {             //NTS-2 (end of file)
            // Backstop exception handler: Catch any exceptions not already 
            // handled elsewhere and try to get debug information from them.
            // RuntimeException objects, in particular, can be thrown by
            // any method without having been declared in a 'throws' clause.
            System.err.println( this + ":" + 
                thrownException + " - " + thrownException.getMessage() ); 
            thrownException.printStackTrace( System.err );
        } 
        
        // Remove the character from the playing field.
        Area charAreaM = environmentI.          // Character's present home.
            getCharacterArea( characterI );  
            
        if( charAreaM != null )                 // Remove character.
            charAreaM.removeCharacter( characterI );
            
        Debug.log("dead");
        
        // Notify environment that the character died.
        environmentI.characterDied( characterI );
    }
    
    /*----------------------------------------------------------------------*\
    ** Interface defined by this class
    \*----------------------------------------------------------------------*/
    
    /** Kill this character.
    */
    public void killCharacter()
    {
        nextStateI = characterDiedS;        // Next state for character.
        interrupt();                        // Abort the delay and handle
                                            //  new character state NOW.
    }
    
    /** Move the character at random.
    * If the character is not found in any area, or is in some area
    * with no neighbors, move the character to some wholly random
    * area. If the character is in an area with neighbors, move it
    * to an adjacent area.
    */
    protected void moveCharacter()
    {
        Area[] targetAreasM = null;             // Places character could go.
        Area charAreaM = environmentI.          // Character's present home.
            getCharacterArea( characterI );        
        
        if( charAreaM != null ) {               // Look for neighboring areas.
            targetAreasM = charAreaM.getNeighbors();
            // Remove foreign character from its old area and repaint area. 
            charAreaM.removeCharacter( characterI );
            gameI.getTheEncounterGame().redisplay( charAreaM );
        }
        
        // Character was nowhere, or charArea had no neighbors.
        // Get whole list of areas and jump at random.
        if( targetAreasM == null || targetAreasM.length == 0 )
            targetAreasM = environmentI.allAreas();
        
        // Assume there's at least one area in the environment.
        Area pickAreaM = targetAreasM[          // Jump to random area.
            (int) Math.floor( Math.random() * targetAreasM.length ) ];
            
        Debug.log( "move to " + pickAreaM );
        
        pickAreaM.addCharacter( characterI );   // Add character to area.
            
        // Show foreign character in its new area.
        EncounterGame.getTheEncounterGame().redisplay( pickAreaM );
    }
    
    /** Generate a random number with the stated average value.
    * For now, use a uniform distribution around the nominal value. 
    *
    * @param    avgP        Average of the randomizer's probability  
    *                       distribution function. 
    * @return               Random number with stated average.
    */
    protected long randAvg(int avgP)
    {
        double randM = Math.random() * avgP * (2 - 2 * RAND_LIMIT);
            
        return Math.round( randM + avgP * RAND_LIMIT );
    }
    
    /** Execute the state machine for the class. Just do the
    * state transition actions, delays are handled elsewhere.
    *
    * <p> Requirement:      SRS paragraph 3.2.FC.3.1, including
    *                       time delay after entry and after
    *                       optimizing quality values to the area.
    *
    * @param    stateP      Carry out the action defined for entry 
    *                       into this state.
    * @return               Next state and amount of time to wait 
    *                       after entering that state. 
    */
    protected StateAndDelay transition( int enteredStateP )
    {
        // If no new state is specified, assume an error and try
        // to recover through a cold start for the character.
        StateAndDelay nextStateM = coldStartS;
        
        switch ( enteredStateP ) {          // Different action for each state.
        case STATE_STARTUP:                     // Finish initialization.
            // More initialization actions go here. 
            // Don't do anything that might affect the display, since that
            // may still be initializing in another thread.
            nextStateM = new StateAndDelay(     // Define next action:
                STATE_JUMP_AREA,                // Jump to a new area.
                randAvg(MS_DELAY_AFTER_JUMP));  // Wait after jumping.
            break;
            
        case STATE_JUMP_AREA: {
            moveCharacter();                // Move the character.
            
            nextStateM = new StateAndDelay(         // Define next action:
                STATE_OPT_AREA,                     // Optimize to the area.
                randAvg( MS_DELAY_AFTER_OPT ) );    // Stay in that state a while.
            break;
            }
            
        case STATE_OPT_AREA: {
            // Find the area containing the character.
            Area charArea = environmentI.getCharacterArea( characterI );
            
            // Redistribute qualities to make the player into the
            // strongest possible competitor for the current area.
            Debug.log("optimize for " + charArea);
            boolean isLive = characterI.
                redistributeQuality( charArea.getAreaQualities() );
                
            // Change of points is a game event. 
            gameI.getEncounterState().handleEvent( characterI );
            
            if ( !isLive )
                nextStateM = characterDiedS;        // Character has died.
            else
                nextStateM = new StateAndDelay(     // Define next action:
                    STATE_JUMP_AREA,                // Jump to a new area.
                    randAvg(MS_DELAY_AFTER_JUMP));  // Wait after jumping.
            break;
            }
            
        case STATE_DEAD:
            nextStateM = characterDiedS;    // Dead characters stay dead.
            break;
            
        default: Debug.fail();              // Should never happen.
        }
        
        return nextStateM;                  // What to do next.
    }
    
    /*----------------------------------------------------------------------*\
    ** java.lang.Object interface
    \*----------------------------------------------------------------------*/

    /** Generate a printable string summarizing the object.
    * Java defines this method as the type-coercion method for
    * converting this object to a String.
    *
    * @return       Printable string representing this object.
    */
    public String toString()
    { 
        return getClass().getName() + "(" + characterI + ")" + super.toString();
    }
}

/* NTS - Notes To Students 
* 1: 'volatile' declaration of nextStateI. This prevents the variable's
*   value from being cached by the Java optimizer. Supposed Java cached the 
*   value in run(), and used the cached copy as the control value
*   for the 'while' loop. If that happened, then nextStateI could not be 
*   set by another thread to change the character state. No matter what the 
*   second thread did, the loop would continue using the stale cached value.
*   A 'volatile' variable is uncacheable; its value has to be read from the 
*   original every time. That means every iteration of the loop checks the 
*   actual value of nextStateI, and detects state change requests made by 
*   other threads. 
* 
* 2:  catch ( Throwable thrownException ) 
*   This is a 'backstop' handler, the last chance for exceptions to be 
*   caught before leaving this application and being handled by the system.
*   Remember that any subclass of RuntimeException may be thrown without
*   being declared in a method's "throws" clause. That include division by
*   zero, null pointers, array bounds checks, etc. Also remember that this 
*   run() method is the top-level control for a thread defined by this
*   application. The backstop handler encloses as much as possible of the
*   thread's logic, giving the programmer a final chance to report or
*   examine unexpected exceptions. It's a good place to set a breakpoint
*   since the "catch" clause should never execute in a correct program.
*/
