package Encounter.EncounterEnvironment;

/* Class Name           : ThumbnailMap
* Version information   : Version 0.1
* Date                  : 02/20/2000
* Copyright Notice      : see below
* Edit history:
*   17 Apr 2000 Tom VanCourt    Simplified data structures after review
*    3 Apr 2000 Tom VanCourt    Filled in a bit.
*    8 Mar 2000 Tom VanCourt    Added setCenterArea
*   20 Feb 2000 Tom VanCourt    Initial coding
*/

/*
    Copyright (C) 2000 Eric J. Braude and Thomas D. Van Court. 
 
    This program is the implementation of the case study specified in 
    "Software Engineering: an Object-Oriented Perspective," Wiley 2001,   
    by Eric J. Braude.

    The program is free software; you can redistribute it and/or modify it 
    under the terms of the GNU General Public License as published by the 
    Free Software Foundation; either version 2 of the License, or any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    For the GNU General Public License, see http://www.gnu.org/copyleft/gpl.txt 
    or write to the Free Software Foundation, Inc., 
    59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Eric Braude can be reached at ebraude@bu.edu and: Boston University,       
    MET Computer Science Department, 808 Commonwealth Avenue, Boston, MA 02215. 
    Tom Van Court can be reached at tvancour@hotmail.com
*/

import java.awt.*;

import FrameworkRPG.Debug;
import TestUtilities.TestExecution;

/** This class handles the thumbnail map of the labyrinth.
* The thumbnail map is the small image of several adjacent Areas
* shown in the lower-right corner of the display.
* <ul title="Design issues:">
* <li> This class is not named in the design spec.
* <li> This map is hard-coded to the layout of the small sample
*   labyrinth. That's a maintenance problem. Some day, this should
*   be rewritten so the thumbnail map gets contructed from the 
*   database that creates the rest of the labyrinth.
* <li> Along with the geometry, connections and names are hard coded.
*   Those are also bugs which need to be fixed.
* <li> The whole labyrinth is small enough to fit completely into
*   the thumbnail map. There is no need (and no code) to change
*   the part of the labyrinth displayed in the thumbnail. Larger
*   labyrinths would eventually need some kind of scrolling, to keep
*   centered on the position of interest, e.g. the player.
* </ul>
*
* <p> Requirement:      implicit in the figure of 3.2.AR.2.2
*
* @author   Tom VanCourt
* @version  0.1
*/

public class ThumbnailMap 
{
    private static class ThRoom {           // Describe a thumbnail room   
        String rmName[];                    // Printable name of room, one
                                            // word at a time. 
        int                                 // Rectangle specified by corners
            x1, y1,                         // upper left
            x2, y2;                         // lower right
        ThRoom(int ix1, int iy1, int ix2, int iy2, String[] iName) {
            x1 = ix1; 
            y1 = iy1; 
            x2 = ix2; 
            y2 = iy2;
            rmName = iName;
        }
    }
    
    private static final int 
        MAP_LETTER_SIZE = 8,                // Point size for map text
        SCALE = 1000,                       // Logical size covered by map
        XUNIT = SCALE/3,                    // Horizontal unit distance
        XHALF = XUNIT/2,                    // Half-step
        YUNIT = SCALE/3,                    // Vertical unit distance
        YHALF = YUNIT/2;                    // Half-step
    
    /** Hard-coded map for the labyrinth as a whole. This should be rewritten
    * some day to construct the map at run time. These data are scaled to a
    * 1000x1000 overall grid. Geometry is rescaled at run time to fit the
    * thumbnail map's display area. 0,0 is the upper-left coordinate.
    */
    private static final ThRoom globalMapS[] = {
        new ThRoom(       0, 1*YUNIT, 1*XUNIT, 3*YUNIT, 
            new String[] {"Dressing", "room"} ),
        new ThRoom( 1*XUNIT,       0, 2*XUNIT, 1*YUNIT, 
            new String[] {"Kitchen"} ),
        new ThRoom( 1*XUNIT, 1*YUNIT, 2*XUNIT, 2*YUNIT, 
            new String[] {"Court", "Yard"} ),
        new ThRoom( 1*XUNIT, 2*YUNIT, 2*XUNIT, 3*YUNIT, 
            new String[] {"Dungeon"} ),
        new ThRoom( 2*XUNIT, 1*YUNIT, 3*XUNIT, 2*YUNIT, 
            new String[] {"Living", "Room"} ),
        new ThRoom( 2*XUNIT, 2*YUNIT, 3*XUNIT, 3*YUNIT, 
            new String[] {"Study"} ),
    };
    
    /** Positions of doors. 
    * If there are the same number of doors as rooms, that's coincidence.
    * There may be more, fewer, or the number. Note that either the X or Y
    * coordinate (but not both) is on a half-unit boundary. That half does
    * tow things. A) It positions the door midway along the wall, and B) by 
    * choosing X or Y, specifies the horizontal or vertical orientation of 
    * the rectangle that visually represents the door.
    */
    private static final Point doorPosS[] = {
        new Point(   XUNIT + XHALF,   YUNIT ),          // Kitchen-court
        new Point(   XUNIT,           YUNIT + YHALF ),  // Dressing-court
        new Point(   XUNIT,         2*YUNIT + YHALF ),  // Dressing-dungeon
        new Point( 2*XUNIT,           YUNIT + YHALF ),  // Courtyard-living
        new Point( 2*XUNIT,         2*YUNIT + YHALF ),  // Dungeon-study
        new Point( 2*XUNIT + XHALF, 2*YUNIT ),          // Living-study
    };
    
    /** Singleton instance of the thumbnailmap object. */
    private static ThumbnailMap theMapS = new ThumbnailMap();
    
    /** Identify the area on which the thumbnail map is centered. 
    * This isn't used yet.
    */
    private Area centerAreaI = null;
    
    /*----------------------------------------------------------------------*\
    ** Constructors
    \*----------------------------------------------------------------------*/
    
    /** Create the singleton object
    */
    private ThumbnailMap()
        { super(); }
    
    /*----------------------------------------------------------------------*\
    ** Interface defined by this class
    \*----------------------------------------------------------------------*/
    
    /** Draw the thumbnail map object.
    * @param drawAreaP      UI object in which map is drawn.
    */
    public void drawMap( Component drawAreaP )
    {
        Graphics drawingM = drawAreaP.getGraphics();    // Graphics context
        Dimension sizeM = drawAreaP.getSize();          // Window size
        
        // TODO: Draw doors to scale, proportional to sizeM
        int doorHeightM = 9,                            // Size of drawn door
            doorWidthM = 3;
        
        drawingM.setColor( Color.black );               // Map color.
        
        Font newFont = new Font(                        // Thumbnail font
                drawingM.getFont().getName(),                     // Same style,
                Font.PLAIN,                             //  nothing fancy,
                MAP_LETTER_SIZE );                      //  small point size.
                
        drawingM.setFont( newFont );                    // Use that font.
        
        // Render each of the rooms.
        for (int i = 0; i < globalMapS.length; i++) {   // Handle each cell
            int x1 = globalMapS[i].x1 * sizeM.width  / SCALE,
                y1 = globalMapS[i].y1 * sizeM.height / SCALE,
                x2 = globalMapS[i].x2 * sizeM.width  / SCALE,
                y2 = globalMapS[i].y2 * sizeM.height / SCALE;
                
            drawingM.drawRect( x1, y1, x2-x1, y2-y1 );  // Draw area boundary.
            
            drawString( drawingM, globalMapS[i].rmName, // Print the name
                (x1 + x2) / 2, (y1 + y2) / 2 );         //  center the string.
        }
        
        for (int i = 0; i < doorPosS.length; i++) {     // Draw the doors
            int x = doorPosS[i].x * sizeM.width / SCALE,
                y = doorPosS[i].y * sizeM.height / SCALE;
                
            if (doorPosS[i].x % XUNIT == 0) {           // Draw door up/down
                drawingM.fillRect(                      // Solid rectangle
                    x - doorWidthM/2, y - doorHeightM/2,// Center of door
                    doorWidthM, doorHeightM);
            } else {                                    // Draw door across
                drawingM.fillRect(                      // Solid rectangle
                    x - doorHeightM/2, y - doorWidthM/2,// Center of door
                    doorHeightM, doorWidthM);
            }
        }
    }
    
    /** Print the name of an area in the thumbnail map. This method  
    * implicitly uses the font and drawing color already set in the graphics 
    * context. The name is an array of stings to be printed in a column. 
    * The column as a whole is centered on the X,Y point given.
    *
    * @param    drawingP        Graphics context for drawing the string.
    * @param    nameStringP     Name to print.
    * @param    xP              X coordinate of the string's midpoint
    * @param    yP              Y coordinate of the string's midpoint
    */
    private static void drawString(Graphics drawingP,
        String nameStringP[], int xP, int yP)
    {
        FontMetrics metricM =                           // Measure the text.
            drawingP.getFontMetrics();  
            
        // This font arithmetic is only approximate, but visually good enough.
        yP = yP + metricM.getAscent() -             // Adjust for text origin
            nameStringP.length *                    //  and the height
                metricM.getHeight() / 2;            //  of the text block.
            
        // Print each line of the text. 
        for (int i = 0; i < nameStringP.length; i++) { 
            drawingP.drawString( nameStringP[i],    // Center the text block.
                xP - metricM.stringWidth( nameStringP[i] ) / 2, yP );
                
            yP += metricM.getHeight();              // Step to next line.
        }
    }
        
    /** Set the area to be displayed at the center of the thumbnail map.
    * @param    newCenterP      New area to show at center of thumbnail.
    */
    public void setCenterArea( Area newCenterP )
    {
        centerAreaI = newCenterP;
    }
   
    /** Return the singleton object. 
    * @return       The ThumbnailMap singleton.
    */
    public static ThumbnailMap getMap()
        { return theMapS; }

    /*----------------------------------------------------------------------*\
    ** Test interface
    \*----------------------------------------------------------------------*/

    /** To test this class.
    *   @param  argsP   One string: file name for test log. 
    *                   Use default log file if no name provided.
    */
    public static void main( String[] argsP ) 
    {
        boolean openedM = false;                    // Assume no log file.
        
        if ( argsP != null && argsP.length > 0 ) {  // If log file is  
            TestExecution.openTestLog( argsP[0] );  // specified, open it.
            openedM = true;
        }
        
        TestExecution.printTitle( "ThumbnailMap" );
        
        // Fill in the tests
            
        if ( openedM )                              // If we opened a file,
            TestExecution.closeTestLog();           // close it. 
    }
}
