package Encounter.EncounterEnvironment;

/* Class Name           : Hyperlink
*  Version information  : Version 0.1
*  Date                 : 02/09/2000
*  Copyright Notice     : see below
*  Edit history:
*    4 Apr 2000     Tom VanCourt    Draw background blocks under captions
*                                   to ensure legibility of caption against
*                                   any area background. Added highlighting
*                                   to drawLink.
*   20 Feb 2000     Tom VanCourt    Added link name, drawing, and testing.
*    9 Feb 2000     Tom VanCourt    Original coding.
*/

/*
    Copyright (C) 2000 Eric J. Braude and Thomas D. Van Court. 
 
    This program is the implementation of the case study specified in 
    "Software Engineering: an Object-Oriented Perspective," Wiley 2001,   
    by Eric J. Braude.

    The program is free software; you can redistribute it and/or modify it 
    under the terms of the GNU General Public License as published by the 
    Free Software Foundation; either version 2 of the License, or any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    For the GNU General Public License, see http://www.gnu.org/copyleft/gpl.txt 
    or write to the Free Software Foundation, Inc., 
    59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Eric Braude can be reached at ebraude@bu.edu and: Boston University,       
    MET Computer Science Department, 808 Commonwealth Avenue, Boston, MA 02215. 
    Tom Van Court can be reached at tvancour@hotmail.com
*/

import java.awt.*;

import FrameworkRPG.Debug;
import TestUtilities.*;

/** Appearance, in an area, of a connection from one area to another.
* <p>Requirements: SRS 3.2.1.2 <i>Travel</i> use case.
* <br> SRS 3.2.CH Connection Hyperlinks between areas.
*
* @author   Tom VanCourt
* @version  0.1
* @see EncounterEnvironmentFactory
*/

public class Hyperlink
{
    /** Size of border around the caption. */
    private static final int CAPTION_BORDER = 6;
    
    /** Default color for the name of the link. */
    private static final Color TEXT_COLOR = Color.red.darker();
    
    /* Range of colors at border of a caption box. */
    private static final Color BORDER_OUTER = Color.black,
        BORDER_INNER = Color.white,
        BORDER_HILIT = Color.yellow;
    
    /** Gradation of colors through caption border. 
    * Static creation of the graded colors avoids the need to interpolate
    * colors, with Color object creation and deletion, on the fly.
    * Use a different series when the caption is highlighted.
    */
    private static Color borderColorS[] = new Color[CAPTION_BORDER],
        borderHilitS[] = new Color[CAPTION_BORDER];
    { 
        int maxStep = borderColorS.length-1;
        
        // Create a series of graded colors from the outer to inner
        // edge of the box around a caption. Use simple linear 
        // interpolation in R, G, and B. Ditto for highlighted boxes.
        for (int i = 0; i <= maxStep; i++) {
            borderColorS[i] = new Color(
                (BORDER_OUTER.getRed() * (maxStep-i) + 
                    BORDER_INNER.getRed() * i) / maxStep,
                (BORDER_OUTER.getGreen() * (maxStep-i) + 
                    BORDER_INNER.getGreen() * i) / maxStep,
                (BORDER_OUTER.getBlue() * (maxStep-i) + 
                    BORDER_INNER.getBlue() * i) / maxStep );
                    
            borderHilitS[i] = new Color(
                (BORDER_OUTER.getRed() * (maxStep-i) + 
                    BORDER_HILIT.getRed() * i) / maxStep,
                (BORDER_OUTER.getGreen() * (maxStep-i) + 
                    BORDER_HILIT.getGreen() * i) / maxStep,
                (BORDER_OUTER.getBlue() * (maxStep-i) + 
                    BORDER_HILIT.getBlue() * i) / maxStep );
        };
    }
    
    /** Debug aid: assign a unique ID value to each link */
    private static int linkSerialNumS = 0;
    
    /** Debug aid: unique serial number assigned to this link. */
    private final int linkSerialNumI = ++linkSerialNumS;
    
    /** Connection that this hyperlink represents. 
    * <p> Requirement 3.2.CH.1
    */
    protected EncounterAreaConnection connectionI = null;
    
    /** ID string shown to the user. */
    private String nameI;
    
    /*----------------------------------------------------------------------*\
    ** Constructors
    \*----------------------------------------------------------------------*/

    /** Construct a basic hyperlink.
    * @param    connectionP     Connection that this hyperlink represents.
    * @param    nameP           Name string printed to represent the link.
    */
    protected Hyperlink( EncounterAreaConnection connectionP, String nameP )
    {
        connectionI = connectionP;
        nameI = nameP;
    }
    
    /*----------------------------------------------------------------------*\
    ** Interface defined by this class
    \*----------------------------------------------------------------------*/

    /** Draw the visible image of the link.
    * Just prints the name. Not fancy, but it's meant to be over-ridden.
    *
    * Maybe someday these captions could be cached as bitmaps for better
    * refresh performance. Not worth the effort at this point.
    *
    * @param    drawAreaP       ValidUI object in which to render this link.
    * @param    highlightedP    <tt>true</tt> to use highlight color
    * @param    whichWallP      Wall against which to render this link.
    *                           Must be some Area.WALL* value.
    * @parap    xP              Coordinate at which to render this link.
    * @param    yP              Coordinate at which to render this link.
    */
    public void drawLink( Component drawAreaP, 
        boolean highlightedP,
        int whichWallP, 
        int xP, int yP )
    {
        Graphics drawContextM = drawAreaP.getGraphics();
        String nameM = getName();                   // Link name string. 
        FontMetrics fontMeasureM = drawContextM.    // Fetch the measurement
            getFontMetrics();                       //  object.
                    
        int                                         // Center the text.
            heightM = fontMeasureM.getHeight(),
            widthM = fontMeasureM.stringWidth( nameM ),
            xPosM = xP - widthM / 2,
            yPosM = yP + heightM / 2;
        
        switch (whichWallP) {                       // Justify to specific wall.
        case Area.WALL_NORTH:
            yPosM = yP + 2*heightM;                 // Down from top.
            break;
        case Area.WALL_SOUTH:
            yPosM = yP - heightM;                   // Up from bottom.
            break;
        case Area.WALL_EAST:
            xPosM = xP - widthM - 2*CAPTION_BORDER; // Abut to right side
            break;
        case Area.WALL_WEST:
            xPosM =  0;                             // Abut to left side.
            break;
        default: Debug.fail();                      // Check valid wall value.
        }
        
        Color outlineM[] = borderColorS;            // Caption box border
            
        if (highlightedP)                           // Highlighted boxes
            outlineM = borderHilitS;                //  use different colors.
        
        for (int i = 0; i < CAPTION_BORDER; i++) {  // Draw the caption box. 
            drawContextM.setColor(outlineM[i]);     // Graded colors in border.
            drawContextM.fillRoundRect(
                xPosM + i, yPosM - heightM + i,     // Upper left corner.
                widthM + 2*(CAPTION_BORDER - i),    // width
                heightM + 2*(CAPTION_BORDER - i),   // height
                CAPTION_BORDER-i+4, CAPTION_BORDER-i+4 );
        }
        
        drawContextM.setColor(TEXT_COLOR);          // Color for writing name.
        drawContextM.drawString( nameM,             // Write the name. 
            xPosM + CAPTION_BORDER, yPosM+3 );      // Cosmetic fudge factors.
    }
    
    /** Fetch the Area at the far end of this link's connection.
    * @param    nearEndP        The origin end of the connection.
    * @return                   The destination end of the connection.
    */
    public Area getOtherArea( Area nearEndP )
        { return (Area) connectionI.getOtherGameArea( nearEndP ); }
    
    /** Fetch a printable string summarizing the hyperlink.
    * @return   User-oriented string describing the link.
    */
    public String getName()
    {
        return nameI;
    }
    
    /*----------------------------------------------------------------------*\
    ** Test interface
    \*----------------------------------------------------------------------*/

    /** To test all registered classes.
    * @param  argsP     One string: file name for test log. 
    *                   Use default log file if no name provided.
    */
    public static void main( String[] argsP ) 
    {
        boolean openedM = false;                    // Assume no log file.
        
        if ( argsP != null && argsP.length > 0 ) {  // If log file is specified, 
            TestExecution.openTestLog( argsP[0] );  // then open it.
            openedM = true;
        }
        
        // Title the following enties in the test log.
        TestExecution.printTitle( "Encounter.EncounterEnvironment.Hyperlink" );
        
        // fill in tests here
        
        if ( openedM )                              // If we opened a file,
            TestExecution.closeTestLog();           // close it. 
    }

    /*----------------------------------------------------------------------*\
    ** java.Object interface
    \*----------------------------------------------------------------------*/

    /** Generate a printable string summarizing the object.
    * Java defines this method as the type-coercion method for
    * converting this object to a String.
    *
    * @return       Printable string representing this object.
    */
    public String toString()
    { 
        return getClass().getName() + "." 
            + linkSerialNumI + " "  
            +  getName(); 
    }
}
