package Encounter.EncounterEnvironment;

/* Class Name           : EncounterEnvironment
* Version information   : Version 0.1
* Date                  : 07/02/1999
* Copyright Notice      : see below
* Edit history:
*   13 Mar 2000     Tom VanCourt    Added getFirstArea
*    6 Mar 2000     Tom VanCourt    Incorporated review comments.
*                                   Added characterDied method.
*    1 Mar 2000     Tom VanCourt    Added allAreas, to help with moving
*                                   the foreign character.
*    8 Feb 2000     Tom VanCourt    Switched to non-static form of Singleton  
*                                   design pattern. Simplified constructors. 
*                                   Added hyperlinks.
*   08 Jan 2000     Tom VanCourt    Change to meet coding standards.
*/

/*
    Copyright (C) 2000 Eric J. Braude and Thomas D. Van Court. 
 
    This program is the implementation of the case study specified in 
    "Software Engineering: an Object-Oriented Perspective," Wiley 2001,   
    by Eric J. Braude.

    The program is free software; you can redistribute it and/or modify it 
    under the terms of the GNU General Public License as published by the 
    Free Software Foundation; either version 2 of the License, or any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    For the GNU General Public License, see http://www.gnu.org/copyleft/gpl.txt 
    or write to the Free Software Foundation, Inc., 
    59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Eric Braude can be reached at ebraude@bu.edu and: Boston University,       
    MET Computer Science Department, 808 Commonwealth Avenue, Boston, MA 02215. 
    Tom Van Court can be reached at tvancour@hotmail.com
*/

import java.util.*;

import Encounter.EncounterCharacters.EncounterCharacter;
import Encounter.EncounterGame.EncounterGame;
import FrameworkRPG.Debug;
import TestUtilities.*;

/** Facade class for this package.
* <p> Design: SDD 3.1 Module decomposition, 
* <br> SDD 4.1 Inter-module dependencies
* <br> SDD 5.1.3 Interface to EncounterEnvironment package
*
* <p> Design issues: <ul>
* <li> Some requirements 3.2.AR.2 are implemented in this class.
* <li> SDD 4.1 says that this class references EncounterCast. No
*   reason was found for doing so.
* <li> Some methods defined in SDD 5.1.3 were not implemented:
*   getArea (replaced internally by hashtable reference), 
*   getAreaConnection (not needed), movePlayerTo (implemented in
*   EncounterGame), moveForeignCharacterTo (implemented differently),
*   and getNeighborhoodAreas (skipped until thumbnail map becomes
*   a more active kind of object).
* </ul>
*
* @author   Dr. Eric Braude
* @version  0.1
*/

public class EncounterEnvironment
{
    /** Minor extension to hashtable. Assumes String indices
    * and assumes that all entries are Area objects. Saves
    * caller from having to typecast every object pulled out of the
    * table since typecasting happens one place: here.
    */
    private class AreaTable extends Hashtable {
        AreaTable()  
            { super(); }
        
        Area get(String areaNameP)
            { return (Area) super.get(areaNameP); }
    }
    
    /** Factory used in this instance. 
    * This is overkill for now, but makes it easier for future updates
    * to change all labyrinth objects at once by changing the factory.
    */
    protected EncounterEnvironmentFactory concreteFactoryI = 
        new EncounterEnvironmentFactory();

    /** The areas in Encounter */
    private AreaTable areasI = new AreaTable();
    
    /** Number of areas in this environment */
    protected int numAreasI;

    /** The unique facade object. 
    * <p> Design SDD 5.1.3
    */
    protected static EncounterEnvironment encounterEnvironmentS;
    
    /* Names of game areas. 
    * Requirements SRS 3.2.AR.2.*
    */
    protected static final String
        NAME_COURTYARD =        "Courtyard",
        NAME_DRESSING_ROOM =    "Dressing room",
        NAME_DUNGEON =          "Dungeon",
        NAME_KITCHEN =          "Kitchen",
        NAME_LIVING_ROOM =      "Living room",
        NAME_STUDY =            "Study";
        
    /** Name of the room in which the player starts the game. */
    protected static final String NAME_START_PLAYER = NAME_DRESSING_ROOM;

    /*----------------------------------------------------------------------*\
    ** Constructors
    \*----------------------------------------------------------------------*/

    /** Construct the labyrinth.
    * <p>Requirements: 3.2.AR.2 names, required qualities, list of areas.
    * <br> 3.2.CH.2 Connection Hyperlink Entities
    * <br> 3.2.CO.2 Connections entities
    */
    protected EncounterEnvironment(EncounterGame gameP)
    {
        Area[] areasM = {                           // Areas in the labyrinth.
            concreteFactoryI.makeArea( NAME_COURTYARD, 
                "Courtyard.jpg", gameP,
                new String[] {EncounterCharacter.QUAL_STAMINA, 
                    EncounterCharacter.QUAL_STRENGTH} ),
            concreteFactoryI.makeArea( NAME_DRESSING_ROOM, 
                "DressingRoom.jpg", gameP,
                new String[] {}, 1, 2 ),
            concreteFactoryI.makeArea( NAME_DUNGEON, 
                "Dungeon.jpg", gameP,
                new String[] {EncounterCharacter.QUAL_PATIENCE, 
                    EncounterCharacter.QUAL_STAMINA} ),
            concreteFactoryI.makeArea( NAME_KITCHEN, 
                "Kitchen.jpg", gameP,
                new String[] {EncounterCharacter.QUAL_CONCENTRATION} ),
            concreteFactoryI.makeArea( NAME_LIVING_ROOM, 
                "LivingRoom.jpg", gameP,
                new String[] {EncounterCharacter.QUAL_CONCENTRATION, 
                    EncounterCharacter.QUAL_STAMINA} ),
            concreteFactoryI.makeArea( NAME_STUDY, 
                "Study.jpg", gameP,
                new String[] {EncounterCharacter.QUAL_CONCENTRATION} ),
        };
        
        areasI.clear();                             // Empty the labyrinth.
        numAreasI = areasM.length;                  // Number of areas.
        for (int i = 0; i < areasM.length; i++)     // Load the area table.
            areasI.put( areasM[i].getName(), areasM[i] );
            
        // Create connections and hyperlinks.
        insertConnection( NAME_COURTYARD,     NAME_DRESSING_ROOM, 
            Area.WALL_WEST );
        insertConnection( NAME_COURTYARD,     NAME_KITCHEN,
            Area.WALL_NORTH );
        insertConnection( NAME_COURTYARD,     NAME_LIVING_ROOM,
            Area.WALL_EAST );
        insertConnection( NAME_DRESSING_ROOM, NAME_DUNGEON,
            Area.WALL_EAST );
        insertConnection( NAME_DUNGEON,       NAME_STUDY,
            Area.WALL_EAST );
        insertConnection( NAME_LIVING_ROOM,   NAME_STUDY,
            Area.WALL_SOUTH );
    }

    /*----------------------------------------------------------------------*\
    ** Interface defined by this class
    \*----------------------------------------------------------------------*/
    
    /** Return a list of all areas.
    * @return       <tt>null</tt> if no areas are known. Array of areas
    *               if there are any at all.
    */
    public synchronized Area[] allAreas()
    {
        Area[] allAreasM = null;
        
        if (numAreasI != 0) {                       // Are there any areas?
            allAreasM = new Area[numAreasI];        // If so, allocate storage.
            Enumeration areaEnumM = areasI.elements();  // Enumerate areas
        
            // Look though whole list of areas.
            for (int i = 0; areaEnumM.hasMoreElements(); i++ )
                allAreasM[i] = (Area)areaEnumM.nextElement();
        }
        
        return allAreasM;
    }
    
    /** Note that a character died
    * @param    characP     Character that just passed on.
    */
    public void characterDied (EncounterCharacter characP)
    {
        // TODO: Initial version of game shuts down when
        //      any character dies.
    }
    
    /** Find the area that contains some character
    * @param    characterP      Character to look for in all areas.
    * @return                   Area containing the character, or  
    *                           <tt>null</tt> if none found.
    */
    public synchronized Area getCharacterArea( EncounterCharacter characterP )
    {
        Area charAreaM = null;                      // Area where found.
        Enumeration areaEnumM = areasI.elements();  // Enumerate areas
        
        // Look though whole list of areas for the one containing 
        // the character specified.
        while( charAreaM == null && areaEnumM.hasMoreElements() ) {
            Area namedArea = (Area) areaEnumM.nextElement();
            
            if ( namedArea.holdsCharacter( characterP ) )
                charAreaM = namedArea;
        }
        
        return charAreaM;
    }
    
    /** To retrieve the EncounterEnvironment singleton object.
    * <p> Design: SDD 5.1.3
    *
    * @param    gameP   Game object for which this is the environment
    *
    * @return   The EncounterEnvironment singleton.
    */
    public synchronized static EncounterEnvironment getEncounterEnvironment(
        EncounterGame gameP) 
    { 
        if (encounterEnvironmentS == null)
            encounterEnvironmentS = new EncounterEnvironment(gameP);
            
        return encounterEnvironmentS; 
    }
        
    /** Get the area in which the player starts the game.
    * @return   Non-null area where player starts. 
    */
    public Area getFirstArea()
        { return areasI.get( NAME_START_PLAYER ); }

    /** Create a connection and place hyperlinks in each end area.
    * <p> Requirement: 3.2.CH.2 Connection Hyperlink Entities,
    *   creates matched pairs of hyperlinks
    *
    * @param    linkFromAreaP   First area in which the hyperlink appears.
    * @param    linkToAreaP     Second area connected by the hyperlink.
    * @param    whichWallP      Specifies which side of the Area
    *                           holds the hyperlink.
    */
    private void insertConnection( String linkFromAreaP, 
        String linkToAreaP, int whichWallP )
    {
        Area toAreaM = areasI.get( linkToAreaP ),   // Find each end area. 
            fromAreaM = areasI.get( linkFromAreaP );
            
        EncounterAreaConnection connectionM =       // Make connection.
            concreteFactoryI.makeConnection( toAreaM, fromAreaM );
            
        fromAreaM.setHyperlink(whichWallP,          // Hyperlink to origin.
            concreteFactoryI.makeHyperlink( connectionM, linkToAreaP ) );
            
        toAreaM.setHyperlink(                       // Hyperlink to target.
            Area.wallOpposite( whichWallP ),        // Reverse direction. 
            concreteFactoryI.makeHyperlink( connectionM, linkFromAreaP ) );
    }
    
    /*----------------------------------------------------------------------*\
    ** Test interface
    \*----------------------------------------------------------------------*/

    /** Test of this class
    *
    * @param        argsP   Command line arguments.
    */
    public static void main(String[] argsP) 
    {
        boolean openedM = false;                    // Assume no log file.
        
        if ( argsP != null && argsP.length > 0 ) {  // If log file is specified, 
            TestExecution.openTestLog( argsP[0] );  // then open it.
            openedM = true;
        }
        
        TestExecution.printTitle( "Encounter.EncounterEnvironment."
            + "EncounterEnvironment" );
            
        // Fill in the tests
        
        if ( openedM )                              // If we opened a file,
            TestExecution.closeTestLog();           // close it. 
    }
}
