package Encounter.EncounterEnvironment;

/* Class Name           : Area
*  Version information  : Version 0.1
*  Date                 : 07/02/1999
*  Copyright Notice     : see below
* Edit history:
*   24 Jul 2000     Tom VanCourt    Cached the character image rather than
*                                   recreating it on every repaint.
*   20 May 2000     Tom VanCourt    Added gameI. Changed addCharacter to
*                                   process the charcter motion event.
*   13 May 2000     Tom VanCourt    Added getAllCharacters
*    4 Apr 2000     Tom VanCourt    Watch for mouse close to an exit,
*                                   highlight nearby exit.
*    6 Mar 2000     Tom VanCourt    Incorporated review comments.
*    1 Mar 2000     Tom VanCourt    Added characters-in-area logic:
*                                   charactersInAreaI, numCharactersInAreaI,
*                                   addCharacter, removeCharacter,
*                                   drawCharacters, holdsCharacter,
*                                   getNeighbors, NeighborList,
*                                   getAreaQualities
*   18 Feb 2000     Tom VanCourt    Incorporated inspection comments.
*                                   Started drawing logic.
*    9 Feb 2000     Tom VanCourt    Added quality list and hyperlinks.
*    8 Feb 2000     Tom VanCourt    Better compliance with design patterns.
*   08 Jan 2000     Tom VanCourt    Change to meet coding standards.
*/

/*
    Copyright (C) 2000 Eric J. Braude and Thomas D. Van Court. 
 
    This program is the implementation of the case study specified in 
    "Software Engineering: an Object-Oriented Perspective," Wiley 2001,   
    by Eric J. Braude.

    The program is free software; you can redistribute it and/or modify it 
    under the terms of the GNU General Public License as published by the 
    Free Software Foundation; either version 2 of the License, or any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    For the GNU General Public License, see http://www.gnu.org/copyleft/gpl.txt 
    or write to the Free Software Foundation, Inc., 
    59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Eric Braude can be reached at ebraude@bu.edu and: Boston University,       
    MET Computer Science Department, 808 Commonwealth Avenue, Boston, MA 02215. 
    Tom Van Court can be reached at tvancour@hotmail.com
*/

import java.awt.*;

import Encounter.EncounterCharacters.EncounterCharacter;
import Encounter.EncounterGame.EncounterGame;
import FrameworkRPG.Debug;
import FrameworkRPG.GameEnvironment.*;
import TestUtilities.*;

/** Places in which Encounter characters encounter each other.
* <p>Invariants:    The list of characters in this area is
*                   packed tightly into the low end of the 
*                   charactersInAreaI array.
* <p>Requirements:  SRS 3.2.AR
* <br> SRS 3.2.AR.1.3 Area-specific qualities
* <br> SRS 3.2.CH.3.1 User clicks on a conneciton hyperlink
*
* <p> Design issues: <ul>
* <li> The maximum number of characters in the area at one time
*   is fixed and small in the initial implementation. That
*   limit show up in the list of characters and in the 
*   drawCharacters logic.
* <li> Some requirements are met, at least partially, in other 
*   classes. In particular, 3.2.AR.1.4 could not reasonably have
*   been implemented in this class.
* </ul>
*
* @author   Dr. Eric Braude
* @version  0.1
*/

public class Area extends GameArea
{
    /** Specify one wall of the area */
    public static final int WALL_EAST = 0;
    
    /** Specify one wall of the area */
    public static final int WALL_WEST = 3;
    
    /** Specify one wall of the area */
    public static final int WALL_NORTH = 1;
    
    /** Specify one wall of the area */
    public static final int WALL_SOUTH = 2;
    
    /** List of quality names that apply to this area. 
    * Default is a list of length 0 (none are important).
    * <p> Requirement: 3.2.AR.1.3 Area-specific qualities.
    */
    protected String[] areaQualitiesI = {};
    
    /** Area's east/west width, in unit cell widths. */
    private int xSizeI = 1;
    
    /** Area's north/south length, in unit cell lengths. */
    private int ySizeI = 1;
    
    /** These values are subscripts to hyperlinkI for each wall.
    * These are the lowest unfilled hyperlink slot on that wall.
    * Subscripts are WALL values. 
    */
    private int[] wallLinkPosI = {0,0,0,0};
    
    /** File name of image to paint for this area.
    * If null, there is no image file. 
    */
    private String imageFileI = null;    
    
    /** 0-length initializer. */
    private static final Hyperlink[] noLinkS = {};
    
    /** Hyperlinks out of this area. 
    * First subscript is one of the WALL values. There must valid
    * a Hyperlink[] array for every array element at this level. 
    * <p>Second subscript is the position (west to east or north to south) 
    * along the wall. Array elements may be null, to indicate absence
    * of a Hyperlink object at some position.
    */
    protected Hyperlink[][] hyperlinkI = 
        {noLinkS, noLinkS, noLinkS, noLinkS};
        
    /** Number of hyperlinks out of this area. */
    protected int numHyperlinkI = 0;
        
    /** Temporary restriction: maximum number of characters in the area. */
    private static final int MAX_CHARACTERS_IN_AREA = 2;
    
    /** List of all characters in the area. Has fixed size for now. */
    protected EncounterCharacter[] charactersInAreaI = 
        new EncounterCharacter[MAX_CHARACTERS_IN_AREA];
        
    /** Number of characters in area */
    protected int numCharactersInAreaI = 0;
    
    /** Game in which the area exists */
    private EncounterGame gameI;
    
    /** Displayable rendering of the area. */
    private Image chImageI;
    
    /*----------------------------------------------------------------------*\
    ** Hyperlink wrappes for allLinks
    \*----------------------------------------------------------------------*/
    
    /** Hyperlink wrapper. When allLinks() iterates over the list of 
    * hyperlinks, this carrier presents each link for whatever operation
    * is being performed.
    * <p> This very roughly follows the Strategy design pattern
    * from "Design Patterns", by Gamma et. al. The allLinks method acts
    * as the Context object, the AreaLinkOperation abstract type acts
    * in the Strategy role, and the concretions of AreaLinkOperation
    * (ClosestLink, etc.) act in the ConcreteStrategyX roles.
    */
    private interface AreaLinkOperation {
        /** Accept an operation to be performed on some hyperlink.
        * @param    linkP       link under consideration.
        * @param    whichWallP  wall on which link appears
        * @param    xP          X coordinate within drawing area
        * @param    yP          Y coordinate within drawing area
        */
        public void linkOperation( Hyperlink linkP, Component drawAreaP,
            int whichWallP, int xP, int yP );
    }
    
    /** Find the hyperlink closest to some specified point. 
    * This treats each link as a single point, for purposes of computing
    * distances. That approximation to the hyperlink's actual on-screen
    * area could conceivably cause counter-intuitive results.
    * <p> Requirement: SRS 3.2.CH.3.1 User clicks on a conneciton hyperlink
    */
    private class ClosestLink implements AreaLinkOperation
    {
        private final int MAX_DIST = 150;   // Max distance considered close.
        
        Hyperlink closestI = null;          // Closest hyperlink found.
        int linkXI, linkYI,                 // Closest link location.
            linkWallI;                      // Closest link's wall.
        private int xI, yI,                 // Target (X,Y) location.
            linkDistSqI = MAX_DIST*MAX_DIST;// Distance^2 from target to link.
            
        // Construct an object for finding distance to the specified point.
        ClosestLink( int xP, int yP ) 
            { xI = xP; yI = yP; }
        
        // See if some link is closer than any other so far.
        public void linkOperation( Hyperlink linkP, Component drawAreaP,
            int whichWallP, int xP, int yP )
        {
            int dxM = xP - xI, 
                dyM = yP - yI,
                distSqM = dxM * dxM + dyM * dyM;    // Distance^2
                
            // Find hyperlink closest to the target.
            if (distSqM < linkDistSqI) { 
                closestI = linkP;                   // Record the link, 
                linkDistSqI = distSqM;              //  it's distance^2,
                linkXI = xP;                        //  and it's location.
                linkYI = yP;
            } 
        }
    }
    
    /** Redraw each hyperlink. 
    */
    private static final AreaLinkOperation drawLinkS = 
        new AreaLinkOperation() {
            public void linkOperation( Hyperlink linkP, Component drawAreaP,
                int whichWallP, int xP, int yP )
            {
                linkP.drawLink(drawAreaP, false, whichWallP, xP, yP); 
            }
        }; 
    
    /** Hyperlink wrapper for collecting the list of neighbors. */
    private class NeighborList implements AreaLinkOperation {
        Area fromI;                                 // Hyperlink origin
        Area[] areaListI = new Area[numHyperlinkI]; // Accumulated list of areas
        int nAreas = 0;                             // Number of areas.
        
        // Construct the neighbor collector
        NeighborList(Area fromP) { fromI = fromP; }
        
        // Fetch the list we accumulated.
        Area[] getList()
            { return (nAreas == 0) ? null : areaListI; }
        
        // Accept a link to add to the list.
        public void linkOperation( Hyperlink linkP, Component drawAreaP,
            int whichWallP, int xP, int yP )
        { 
            areaListI[nAreas++] = linkP.getOtherArea(fromI); 
        }
    }
    
    /*----------------------------------------------------------------------*\
    ** Constructors
    \*----------------------------------------------------------------------*/
    
    /** Basic area constructor.
    *
    * @param    aNameP      The name of the area object.
    * @param    gameP       Game in which area exists.
    * @param    qualListP   Non-null. Quality names important in this area.
    * @param    imageFileP  File name of area's background image.
    * @param    xSizeP      X (east/west) dimension, in grid cell widths > 0.
    * @param    ySizeP      Y (north/south) dimension, in grid cell heights > 0.
    */
    protected Area( String aNameP, String imageFileP, 
        EncounterGame gameP, String[] qualListP, 
        int xSizeP, int ySizeP )
    {
        super( aNameP );
        
        gameI = gameP;                              // Remember the game
        
        xSizeI = xSizeP;                            // Note dimensions
        ySizeI = ySizeP;                            // of this area.
        
        imageFileI = imageFileP;                    // Save area info.
        areaQualitiesI = qualListP;
        
        // Each unit distance along each wall may have a hyperlink. 
        // Null array entries represent missing hyperlinks. 
        hyperlinkI[WALL_EAST]  = new Hyperlink[ySizeP];
        hyperlinkI[WALL_WEST]  = new Hyperlink[ySizeP];
        hyperlinkI[WALL_NORTH] = new Hyperlink[xSizeP];
        hyperlinkI[WALL_SOUTH] = new Hyperlink[xSizeP];
    }
    
    /*----------------------------------------------------------------------*\
    ** Interface defined by this class
    \*----------------------------------------------------------------------*/
    
    /** Add a character to this area.
    * This area may or may not be shown at the UI. If it's visible,
    * the caller needs to redraw so the character appears on screen.
    *
    * <p>Requirements:      SRS 3.2.AR.4.2 (characters in same area)
    * <p>Precondition:      addedCharP must be non-null and must not
    *                       duplicate any character already in the area.
    * @param    addedCharP  Character added to the area.
    * @return   <tt>true</tt> if this area end up with more than
    *                       one character.
    */
    public void addCharacter( EncounterCharacter addedCharP )
    {
        // Insert the character at the end of the list.
        synchronized(this) {
            charactersInAreaI[numCharactersInAreaI++] = addedCharP;
        }
        
        // Paint this game area on screen
        gameI.redisplay(this);
        
        // Note the character motion event.
        gameI.getEncounterState().
            handleEvent(addedCharP, this, numCharactersInAreaI > 1);
    }
    
    /** Iterate over hyperlinks in the Area. Assign actual coordinates
    * to each hyperlink and perform some operation.
    *
    * @param    drawAreaP   Java AWT component in which to draw the links.
    * @param    linkOpP     Operation to perform on each link
    */
    synchronized void allLinks( 
        Component drawAreaP, AreaLinkOperation linkOpP )
    {
        Dimension drawSizeM = (drawAreaP != null) ? 
            drawAreaP.getSize() :       // Real size if drawAreaP is valid.
            new Dimension(100, 100);    // Place-holder otherwise.
        
        int nLinksOnSideM = hyperlinkI[WALL_EAST].length;
        for (int i = 0; i < nLinksOnSideM; i++) {
            Hyperlink link = hyperlinkI[WALL_EAST][i];
            if ( link != null )
                linkOpP.linkOperation( link, drawAreaP, 
                    WALL_EAST, drawSizeM.width-1, 
                    (i+1) * drawSizeM.height / (nLinksOnSideM+1) );
        }
        
        nLinksOnSideM = hyperlinkI[WALL_WEST].length;
        for (int i = 0; i < nLinksOnSideM; i++) {
            Hyperlink link = hyperlinkI[WALL_WEST][i];
            if ( link != null )
                linkOpP.linkOperation( link, drawAreaP, WALL_WEST, 0, 
                    (i+1) * drawSizeM.height / (nLinksOnSideM+1) );
        }
        
        nLinksOnSideM = hyperlinkI[WALL_NORTH].length;
        for (int i = 0; i < nLinksOnSideM; i++) {
            Hyperlink link = hyperlinkI[WALL_NORTH][i];
            if ( link != null )
                linkOpP.linkOperation( link, drawAreaP, WALL_NORTH,  
                    (i+1) * drawSizeM.width / (nLinksOnSideM+1),
                    0) ;
        }
        
        nLinksOnSideM = hyperlinkI[WALL_SOUTH].length;
        for (int i = 0; i < nLinksOnSideM; i++) {
            Hyperlink link = hyperlinkI[WALL_SOUTH][i];
            if ( link != null )
                linkOpP.linkOperation( link, drawAreaP, WALL_SOUTH,  
                    (i+1) * drawSizeM.width / (nLinksOnSideM+1),
                    drawSizeM.height-1 );
        }
    }
            
    /** Output the area name.
    */
    public void announce() 
        { Debug.log( this + " noted" ); }
        
    /** Create the on-screen image of the area.
    * This fills the drawing area and paints the area name.
    * If there's a bitmap file, this paints the file instead.
    * <p> Requirements: SRS3.2.AR.4.1 Display on entry of player character.
    * <br> SRS 3.1.1 and 3.2.AR.2 Display of hyperlinks in area
    *
    * @param    drawAreaP   Java AWT component in which to draw the area.
    */
    public void drawArea( Component drawAreaP )
    {
        Graphics drawM = drawAreaP.getGraphics();
        Dimension drawSizeM = drawAreaP.getSize();
        
        if (imageFileI == null) {                   // If there's no image, 
            Color oldColorM = drawM.getColor();     // Save current color. 
            drawM.setColor(                         // Background fill color: 
                new Color( 204, 153, 255 ) );       //  purplish blue.
            drawM.fillRect( 0, 0,drawSizeM.width,   // Rectangle covers 
                drawSizeM.height );                 //  entire background. 
            
            String nameM = getName();               // Area name string. 
            FontMetrics fontMeasureM = drawM.       // Fetch the measurement
                getFontMetrics();                   //  object.
            int nameWidthM = fontMeasureM.          // Pixel width for name. 
                        stringWidth( nameM );
                            
            drawM.setColor( Color.black );          // Pick lettering color.
            drawM.drawString( nameM,                // Print area name,  
                (drawSizeM.width - nameWidthM) / 2, //  centered. 
                (drawSizeM.height - fontMeasureM.getHeight()) / 2 );
            
            drawM.setColor( oldColorM );            // Restore drawing color.
      
        } else {
            // Instantiate image on frst reference. 
            if (chImageI == null)
                chImageI = drawAreaP.getToolkit().getImage( imageFileI );
                
            drawM.drawImage( chImageI,              // Draw the area image. 
                0, 0, drawSizeM.width, drawSizeM.height, drawAreaP );
        }
        
        allLinks( drawAreaP, drawLinkS );   // Render the hyperlinks.
        drawCharacters( drawAreaP );        // Render characters in this area.
    }
    
    /** Draw the characters in this area. This method is hard-coded to
    * the temporary limit of two characters per area. This method is
    * NOT synchronized. Instead, it's private and expects synchronization
    * at the caller's level.
    *
    * @param    drawAreaP   Java AWT component in which to draw the area.
    */
    private void drawCharacters( Component drawAreaP )
    {
        Dimension drawSizeM = drawAreaP.getSize();
        Graphics areaGraphicsM = drawAreaP.getGraphics();
        
        if (numCharactersInAreaI > 0)   // Draw first character if any
            charactersInAreaI[0].showCharacter(
                drawAreaP, areaGraphicsM,
                new Point(drawSizeM.width/4, drawSizeM.height/2),
                drawSizeM.height/2, false );
                
        if (numCharactersInAreaI > 1)   // Draw second character if any
            charactersInAreaI[1].showCharacter(
                drawAreaP, areaGraphicsM,
                new Point(3*drawSizeM.width/4, drawSizeM.height/2),
                drawSizeM.height/2, true );
    }
    
    /** Return the list of characters in this area.
    * @return   list of characters.
    */
    public synchronized EncounterCharacter[] getAllCharacters()
    {
        EncounterCharacter[] allCharM = new
            EncounterCharacter[ numCharactersInAreaI ];
            
        for (int i = 0; i < numCharactersInAreaI; i++)
            allCharM[i] = charactersInAreaI[i];
            
        return allCharM;
    }
    
    /** Return the list of qualities that matter in this area.
    * @return   list of qualities.
    */
    public String[] getAreaQualities()
    {
        // TODO: It's a bit hazardous to expose the array
        // and string literals that define this area's 
        // qualities. It would be safer to reinstantiate
        // the array and strings, then return the new instance.
        // This is good enough for now.
        return areaQualitiesI;
    }
    
    private ClosestLink lastCloseLinkI = null;
    
    /** Return the hyperlink near a mouse event.
    * <p> Requirement: SRS 3.2.CH.3.1 User clicks on a connection 
    *   hyperlink
    * @param    drawAreaP   area in which mouse event happened.
    * @param    xP          X coordinate of mouse
    * @param    yP          Y coordinate of mouse
    * @return   Hyperlink closest to the mouse coordinate. May
    *           be <tt>null</tt> if no link is very close.
    * @see      ClosestLink
    */
    public Hyperlink getHyperlink( Component drawAreaP, int xP, int yP )
    {
        ClosestLink linkFinderM =           // Set up to locate links. 
            new ClosestLink( xP, yP );
            
        allLinks( drawAreaP, linkFinderM ); // Find closest hyperlink.
        
        // Un-highlight the old hyperlink, if there was one.
        synchronized( this ) {
            if (lastCloseLinkI != null && lastCloseLinkI.closestI != null)
                lastCloseLinkI.closestI.drawLink(
                    drawAreaP, false,
                    lastCloseLinkI.linkWallI, 
                    lastCloseLinkI.linkXI, 
                    lastCloseLinkI.linkYI );
        }
        
        lastCloseLinkI = linkFinderM;       // Record the new choice.
        
        synchronized( this ) {
            if (lastCloseLinkI.closestI != null) // Highlight new choice.
                lastCloseLinkI.closestI.drawLink(
                    drawAreaP, true,
                    lastCloseLinkI.linkWallI, 
                    lastCloseLinkI.linkXI, 
                    lastCloseLinkI.linkYI );
        }
        
        return linkFinderM.closestI;        // Return the closest link.
    }
    
    /** Return the list of other areas that abut this area.
    * @return   <tt>null</tt> if there are no neighbors, 
    *           an array of neighbor references otherwise.
    */
    public Area[] getNeighbors()
    {
        NeighborList neighborsM =       // New object to
            new NeighborList(this);     //  collect the list of neighbors.
        allLinks(null, neighborsM);     // Collect the list.
        return neighborsM.getList();    // Report the collection.
    }
    
    /** Determine whether a specified character is in this area.
    *
    * @param    characterP  Character to look for in the area.
    * @return   <tt>true</tt> if this area holds the character.
    */
    public synchronized boolean holdsCharacter( EncounterCharacter characterP )
    {
        boolean foundM = false;
        
        for( int i = 0; i < numCharactersInAreaI && !foundM; i++ )
            if( charactersInAreaI[i] == characterP )
                foundM = true;
        
        return foundM;
    }
        
    /** Remove a character from this area.
    * This area may or may not be shown at the UI. If it's visible,
    * the caller needs to redraw so the character appears on screen.
    *
    * @param    characterP  Character to be removed.
    */
    public synchronized void removeCharacter( EncounterCharacter characterP )
    {
        // Look everywhere in the list for the character.
        // Remove the character wherever it's found. Keep the list
        // packed into the low array indices by moving last in the
        // list into the emptied slot.
        for( int i = 0; i < numCharactersInAreaI; )
            if( charactersInAreaI[i] == characterP )
                charactersInAreaI[i] =              // Remove from list. 
                    charactersInAreaI[--numCharactersInAreaI];
            else 
                i += 1;                             // Work through the list.
    }
    
    /** Add a hyperlink to the area. Redundant settings are not allowed.
    * Links are added to a wall in order, north to south or east to west.
    *
    * @param    wallP       Which wall holds the link.
    * @param    hyperlinkP  Link to be inserted. May be null if there's 
    *                       no link at the next slot along the wall.
    *                       Must be a link involving this Area.
    * @return               Position of the hyperlink along its wall.
    */
    public int setHyperlink( int wallP, Hyperlink hyperlinkP )
    {
        if (hyperlinkP != null)         // Count the hyperlinks. 
            numHyperlinkI += 1;
            
        // Assign the hyperlink to the first unused slot along the wall.
        hyperlinkI[wallP][wallLinkPosI[wallP]] = hyperlinkP;
            
        // Increment the available-slot counter, but return the 
        // slot number at which we just placed the link.
        return wallLinkPosI[wallP]++;
    }
  
  
    /** Reverse a wall direction. 
    * @param    wallP   Wall direction, assumed to be valid.
    * @return           Direction opposite wallP
    */
    public static int wallOpposite( int wallP )
    {
        // This logic depends on the exact values for directions.
        return 3 - wallP;
    }
    
    /*----------------------------------------------------------------------*\
    ** Test interface
    \*----------------------------------------------------------------------*/

    /** To test this class.
    *   @param  argsP   One string: file name for test log. 
    *                   Use default log file if no name provided.
    */
    public static void main( String[] argsP ) 
    {
        boolean openedM = false;                    // Assume no log file.
        
        if ( argsP != null && argsP.length > 0 ) {  // If log file is  
            TestExecution.openTestLog( argsP[0] );  // specified, open it.
            openedM = true;
        }
        
        // Title the following entries in the test log.
        TestExecution.printTitle( "Encounter.EncounterEnvironment.Area" );
            
        batchTests();                               // Do the tests.
        imageTests();                               // Do the graphical tests. 
    
        if ( openedM )                              // If we opened a file,
            TestExecution.closeTestLog();           // close it. 
    }
    
    /** Generate a stripped-down Area object for testing.
    */
    public static Area getTestArea( String nameM )
    {
        return new Area( nameM, null, null, null, 1, 1 );            
    }
    
    /** Run tests on this class.
    */
    private static void batchTests()
    {
        final String dummyNameM = "dummy area";     // Name string
        Area testAreaM = getTestArea( dummyNameM ); // Testable object.
        
        // Test the opposite-wall logic.
        TestExecution.printReportToFile( "Opposites.1",
            wallOpposite(WALL_EAST), WALL_WEST );
        TestExecution.printReportToFile( "Opposites.2",
            wallOpposite(WALL_WEST), WALL_EAST );
        TestExecution.printReportToFile( "Opposites.3",
            wallOpposite(WALL_NORTH), WALL_SOUTH );
        TestExecution.printReportToFile( "Opposites.4",
            wallOpposite(WALL_SOUTH), WALL_NORTH );
            
        // Verify name initialization.
        TestExecution.printReportToFile( "Naming",
            dummyNameM, testAreaM.getName() );
    }

    /** Run graphical tests on this class.
    */
    private static void imageTests()
    {
        Area testAreaM = getTestArea( "Drawn area" ); // Testable object.
        
        Frame testImageM = new testAreaImage( testAreaM );
        testImageM.setSize(400, 250);
        testImageM.setLocation(40, 40);
        testImageM.setVisible(true);
        testImageM.show();
        
        try {
            Thread.currentThread().sleep(15*1000);
        } catch( Exception exc) {
        }
        
        testImageM.dispose();
    }
    
    /** Class to test repainting of characters. Creates a window, which will 
    * contain several copies of the character image.
    */
    private static class testAreaImage extends Frame
    {
        private Area areaI;
        
        testAreaImage( Area areaP )
        {
            super(areaP.getName());
            areaI = areaP;
        }
        
        public void paint( Graphics drawP )
        {
            areaI.drawArea( this );
        }
    }  
}
