package Encounter.EncounterCharacters;

/*Class Name            : PlayerCharacter
* Version information   : Version 0.1
* Date                  : 07/02/1999
* Copyright Notice      : see below
* Edit history:
*    2 May 2000     Tom VanCourt    Added old quality values to support 
*       requirement 3.2.AR.4.6. Added updateBegine() and updateEnd() to 
*       support requirement 3.2.PQ.4.4.                                   
*   19 Feb 2000     Tom VanCourt    Added image file to constructor.
*   08 Jan 2000     Tom VanCourt    Change to meet coding standards.
*/

/*
    Copyright (C) 2000 Eric J. Braude and Thomas D. Van Court. 
 
    This program is the implementation of the case study specified in 
    "Software Engineering: an Object-Oriented Perspective," Wiley 2001,   
    by Eric J. Braude.

    The program is free software; you can redistribute it and/or modify it 
    under the terms of the GNU General Public License as published by the 
    Free Software Foundation; either version 2 of the License, or any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    For the GNU General Public License, see http://www.gnu.org/copyleft/gpl.txt 
    or write to the Free Software Foundation, Inc., 
    59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Eric Braude can be reached at ebraude@bu.edu and: Boston University,       
    MET Computer Science Department, 808 Commonwealth Avenue, Boston, MA 02215. 
    Tom Van Court can be reached at tvancour@hotmail.com
*/

import FrameworkRPG.Debug;
import FrameworkRPG.Characters.*;
import TestUtilities.TestExecution;

/** PlayerCharacter class/object, a specialized EncounterCharacter.
* <p> Requirements: SRS 3.2.PC
* <p> Design issues: <ul>
* <li> SRS 3.2.AR.4.6 made requirements that could not reasonably be
*   implemented in the Area object, at least not completely. This class
*   provides the service used by that requirement, so this class is
*   affected by that requirement. Methods and data related to the
*   player's old quality values support SRS 3.4.AR.4.6, but were not
*   mentioned in SDD 6.1.2.PC.
* <li> It's imaginable that two threads could concurrently access 
*   the player's quality values. One thread could start an update and
*   change some values. A second thread could read the partially changed
*   values and make some decisions based on those data. The first thread
*   could come back later roll back the quality values to what they were
*   before the change began. The second is now stuck with decisions based
*   on incorrect quality values. The game as of this writing (11 May 2000)
*   does not have any such circumstance in it, but future changes could
*   bring that concurrency bug to life.
* </ul>
* @author   Dr. Eric Braude
* @version  0.1
*/

public class PlayerCharacter extends EncounterCharacter 
{
    /** The user-operated character. */
    private static PlayerCharacter mainPlayerCharacterS;
    
    /** Quality values prior to tentative update.
    * <p> Supports requirement 3.2.PQ.4.4
    */
    private float[] preUpdateValueI = new float[ qualValueI.length ];
    
    /** True when an update has been started but not completed.
    * <p> Supports requirement 3.2.PQ.4.4
    */
    private boolean updateInProgressI = false;
    
    /*----------------------------------------------------------------------*\
    ** Constructors
    \*----------------------------------------------------------------------*/

    /** Constructor 
    *
    * @param        aNameP      String Name of new player.
    * @param        aImageFileP Name of image file. 
    */
    protected PlayerCharacter( String aNameP, String aImageFileP ) 
    { 
        super(aNameP, aImageFileP);
    }

    /*----------------------------------------------------------------------*\
    ** Interface defined by this class
    \*----------------------------------------------------------------------*/

    /** Creates a player character automatically adding it to the list 
    * of characters.
    *
    * @param        aNameP      String Name of new player.
    * @param        aImageFileP Name of image file. 
    * @return                   The new player character.
    */
    public static PlayerCharacter getMainPlayerCharacter(
        String aNameP, String aImageFileP ) 
    {
        /* Check to see if the character was created previously */
        if (mainPlayerCharacterS == null) {
            mainPlayerCharacterS = new PlayerCharacter(aNameP, aImageFileP);

            Debug.log(mainPlayerCharacterS + " created.");
        }
        
        return mainPlayerCharacterS;
    }
    
    /** Note that an update is beginning and save the quality values.
    * <p>Requirement: 3.2.PQ.4.4
    */
    public synchronized void updateBegin()
    {
        // Updates should not overlap.
        Debug.assert( !updateInProgressI, "Overlapping updates" );
        
        updateInProgressI = true;       // Update is now in progress. 
        
        for ( int i = 0; i < qualValueI.length; i++ )
            preUpdateValueI[i] = qualValueI[i];
    }

    /** Note that an update is ending. 
    * <p>Requirement: 3.2.PQ.4.4
    * @param    commitP     If <tt>true</tt>, roll forward to the updated
    *                       quality values. If <tt>false</tt>, roll back to 
    *                       values in effect before the update started.
    */
    public synchronized void updateEnd(boolean commitP)
    {
        // If we're stopping an update, there should be an update.
        Debug.assert( updateInProgressI, "Non-update ends" );
        
        // If the update ends but does not commit to the new values,
        // revert to values in effect before the update started.
        if ( !commitP )
            for ( int i = 0; i < qualValueI.length; i++ )
                qualValueI[i] = preUpdateValueI[i];
                
        updateInProgressI = false;      // Update is now completed. 
    }

    /*----------------------------------------------------------------------*\
    ** Test interface
    \*----------------------------------------------------------------------*/

    /** To test this class.
    *   @param  argsP   One string: file name for test log. 
    *                   Use default log file if no name provided.
    */
    public static void main( String[] argsP ) 
    {
        boolean openedM = false;                    // Assume there's no log.
        
        if ( argsP != null && argsP.length > 0 ) {  // If log file is specified, 
            TestExecution.openTestLog( argsP[0] );  // then open it.
            openedM = true;
        }
        
        TestExecution.printTitle(                   // Title the log entries.
            "Encounter.EncounterCharacters.PlayerCharacter" );  
            
        /* Fill in tests here. */
    
        if ( openedM )                              // If we opened a file,
            TestExecution.closeTestLog();           // close it. 
    }  
}
