package Encounter.EncounterCharacters;

/* Class Name           : ForeignCharacter
* Version information   : Version 0.1
* Date                  : 07/02/1999
* Copyright Notice      : see below
* Edit history:
*   21 Mar 2000     Tom VanCourt    Allow quality point redistribution
*                       for areas with zero-length quality lists, e.g. 
*                       the dressing room.
*    6 Mar 2000     Tom VanCourt    Incorporated review comments
*    1 Mar 2000     Tom VanCourt    Added redistributeQuality
*                                   and matching tests.
*   19 Feb 2000     Tom VanCourt    Added image file to constructor.
*   11 Feb 2000     Tom VanCourt    Added main() test entry point
*   08 Jan 2000     Tom VanCourt    Change to meet coding standards.
*/

/*
    Copyright (C) 2000 Eric J. Braude and Thomas D. Van Court. 
 
    This program is the implementation of the case study specified in 
    "Software Engineering: an Object-Oriented Perspective," Wiley 2001,   
    by Eric J. Braude.

    The program is free software; you can redistribute it and/or modify it 
    under the terms of the GNU General Public License as published by the 
    Free Software Foundation; either version 2 of the License, or any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    For the GNU General Public License, see http://www.gnu.org/copyleft/gpl.txt 
    or write to the Free Software Foundation, Inc., 
    59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Eric Braude can be reached at ebraude@bu.edu and: Boston University,       
    MET Computer Science Department, 808 Commonwealth Avenue, Boston, MA 02215. 
    Tom Van Court can be reached at tvancour@hotmail.com
*/

import FrameworkRPG.Characters.*;
import FrameworkRPG.Debug;
import TestUtilities.TestExecution;

/** ForeignCharacter class/object, a specialized EncounterCharacter.
* <p> Requirements: SRS 3.2.FC
* <p> Design issues: <ul>
* <li> The redistributeQuality method could be optimized for less redundant 
*   searching and for fewer new temporary objects.
* </ul>
*
* @author   Dr. Eric Braude
* @version  0.1
*/

public class ForeignCharacter extends EncounterCharacter 
{
    /** Singleton character object. */
    private static ForeignCharacter mainForeignS;
    
    /*----------------------------------------------------------------------*\
    ** Constructors
    \*----------------------------------------------------------------------*/

    /** Constructor 
    *
    * @param        NameP      String Name of new player.
    * @param        ImageFileP Name of image file. 
    */
    protected ForeignCharacter( String NameP, String ImageFileP ) 
        { super(NameP, ImageFileP); }

    /*----------------------------------------------------------------------*\
    ** Interface defined by this class
    \*----------------------------------------------------------------------*/

    /** To create a foreign character automatically adding it to the
    *  list of characters.
    *
    * @param        aNameP      The name of the foreign character.
    * @param        aImageFileP Name of image file. 
    * @return                   New foreign character.
    */
    public static ForeignCharacter getForeignCharacter( 
        String NameP, String ImageFileP ) 
    {
        if ( mainForeignS == null ) {
            mainForeignS = new ForeignCharacter( NameP, ImageFileP );
            Debug.log( mainForeignS + " created." );
        }
        
        return mainForeignS;
    }
    
    /** Redistribute the character's quality points so as to make the
    * character the strongest possible competitor for the
    * list of qualities given. This method needs to be 
    * <tt>synchronized</tt>, since intermediate results put the list
    * of quality values through temporary invalid states.
    *
    * <p>Requirement:       SRS 3.2.FC.3.1 
    * <p>Preconditions:     qualTargetP is non-null,
    *                       contains only valid quality names, and does
    *                       not repeat any names. 
    * <p>Postconditions:    The quality point total after this method is
    *                       equal to the total before, unless the point 
    *                       values get truncated to 0.
    * @param    qualTargetP List of quality names to be given the
    *                       character's whole quality total.
    * @return   <tt>true</tt> iff the character's point total is over 0.
    */
		public synchronized boolean redistributeQuality( String[] qualTargetP )
    {
        Debug.log("redist: n qual = " + qualTargetP.length);
                
        if (qualTargetP.length != 0) {
            float redistValueM = sumOfQualities()   // Total quality points
                / qualTargetP.length;               // spread across list.
                
            String[] qualListM = getQualityTypes(); // List of quality names.
            
            for (int i = 0; i < qualListM.length; i++)
                setQuality( qualListM[i], 0.0F );   // Clear the old values.
                
            for (int i = 0; i < qualTargetP.length; i++)    // Set new values.
                setQuality( qualTargetP[i], redistValueM );
        }
        
        return sumOfQualities() > 0.0F;             // Is character alive?
    }

    /*----------------------------------------------------------------------*\
    ** Test interface
    \*----------------------------------------------------------------------*/

    /** To test this class.
    *   @param  argsP   One string: file name for test log. 
    *                   Use default log file if no name provided.
    */
    public static void main( String[] argsP ) 
    {
        boolean openedM = false;                    // Assume we don't open a log
        
        if ( argsP != null && argsP.length > 0 ) {  // If log file is specified, 
            TestExecution.openTestLog( argsP[0] );  // then open it.
            openedM = true;
        }
        
        TestExecution.printTitle(                   // Title the log entries.
            "Encounter.EncounterCharacters.ForeignCharacter" );  
            
        // redistributeQuality tests.
        ForeignCharacter testGuy = new ForeignCharacter("alpha", null);
        float initTotalM = testGuy.sumOfQualities();
        
        // Turn all quality points into strength, check results.
        testGuy.redistributeQuality(
            new String[] {QUAL_STRENGTH} );
        TestExecution.printReportToFile( "redistributeQuality 1.1",
            initTotalM, testGuy.getQualityValue(QUAL_STRENGTH) );
        TestExecution.printReportToFile( "redistributeQuality 1.2",
            0.0F, testGuy.getQualityValue(QUAL_CONCENTRATION) );
        
        // Split all points between two qualities and check results.
        testGuy.redistributeQuality(
            new String[] {QUAL_STRENGTH, QUAL_STAMINA} );
        TestExecution.printReportToFile( "redistributeQuality 2.1",
            initTotalM / 2, testGuy.getQualityValue(QUAL_STRENGTH) );
        TestExecution.printReportToFile( "redistributeQuality 2.2",
            0.0F, testGuy.getQualityValue(QUAL_CONCENTRATION) );
    
        if ( openedM )                              // If we opened a file,
            TestExecution.closeTestLog();           // close it. 
    }  
}
