package Encounter.EncounterCharacters;

/* Class Name:      EncounterCharacter
* Date:             01/13/2000
* Copyright Notice: see below
* Edit history:
*   24 Jul 2000 Tom VanCourt    Cached character image rather than
*                               recreating it on every repaint.
*   13 May 2000 Tom VanCourt    Moved saveQualityValues, getOldValue
*                               and oldValueI from PlayerCharacter
*/

/*
    Copyright (C) 2000 Eric J. Braude and Thomas D. Van Court. 
 
    This program is the implementation of the case study specified in 
    "Software Engineering: an Object-Oriented Perspective," Wiley 2001,   
    by Eric J. Braude.

    The program is free software; you can redistribute it and/or modify it 
    under the terms of the GNU General Public License as published by the 
    Free Software Foundation; either version 2 of the License, or any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    For the GNU General Public License, see http://www.gnu.org/copyleft/gpl.txt 
    or write to the Free Software Foundation, Inc., 
    59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Eric Braude can be reached at ebraude@bu.edu and: Boston University,       
    MET Computer Science Department, 808 Commonwealth Avenue, Boston, MA 02215. 
    Tom Van Court can be reached at tvancour@hotmail.com
*/

import java.awt.*;
import java.io.*;

import FrameworkRPG.Characters.*;
import FrameworkRPG.Debug;
import TestUtilities.*;

/** Base class for the characters of the Encounter game.
* <p> Requirements: SRS 3.2.EC
* <p> Design: SDD 6.2.1
* <p> Invariants: <ol>
* <li> encounterCharacterS contains all objects of this class
* <li> The values of qualValueI[] are >= 0
* </ol>
* <p> Design issues: <ul>
* <li> Character images are switched freely between right- and 
*   left-handed forms. That means that sword fighters will switch
*   hands; scabbards will swing back and forth; scars, tattoos, etc.
*   will move around, hair styles will flop back and forth, etc. 
*   It's good enough for now, though.
* </ul>
* @author  Eric Braude, Tom VanCourt
* @version 0.2
*/

public class EncounterCharacter extends GameCharacter
{
    /** Total quality points at initialization. 
    * <p> Requirement 3.2.EC.1.2
    */
    public static final float QUAL_TOTAL_INIT = 100.0f;

    // Symbols used when other classes refer to specific qualities.

    /** Symbol for one of a character's qualities  */
    public static final String QUAL_CONCENTRATION = "concentration";

    /** Symbol for one of a character's qualities */
    public static final String QUAL_INTELLIGENCE = "intelligence";

    /** Symbol for one of a character's qualities */
    public static final String QUAL_PATIENCE = "patience";

    /** Symbol for one of a character's qualities */
    public static final String QUAL_STAMINA = "stamina";

    /** Symbol for one of a character's qualities */
    public static final String QUAL_STRENGTH = "strength";

    /** Qualities that each Encounter character posesses. 
    * <p>Req: 3.2.EC.1.2 
    */
    protected static final String[] qualityTypeS =
    {    QUAL_CONCENTRATION,         QUAL_STAMINA,
         QUAL_INTELLIGENCE,          QUAL_PATIENCE,
         QUAL_STRENGTH
    };

    /** Root directory for graphics files. */
    static final String FILE_ROOT = "Graphics/";

    /* INSTANCE VARIABLES */
    /** Values of the qualities . <p> Requirement 3.2.EC.1.2 */
    protected float[] qualValueI = new float[ qualityTypeS.length ];

    /** Quality values before the last engagement
    * <p> Requirement:  3.2.AR.4.6
    */
    protected float[] oldValueI = new float[ qualValueI.length ];

    /** Name of GIF file containing the character image.
    * The character in this image is assumed to be facing left.
    * Select this character's height, relative to heights of other characters,
    *   by padding the top and bottom with transparent pixels. No padding gives
    *   the tallest possible character.
    */
    protected String imageFileNameI = null;
    
    /** Displayable image for the character. Initialized when used. */
    private Image chImageI = null;

    /* CONSTRUCTORS */

    /** Allocate initial total quality points equally among the qualities.
    * <p> Requirement: 3.2.EC.1.2 (quality value initialization)
    */
    protected EncounterCharacter()
    {   super();
        for( int i = 0; i < qualityTypeS.length; ++i )
            qualValueI[i] = QUAL_TOTAL_INIT / qualityTypeS.length;

        saveQualityValues();
    }

    /** Construct a new character using the given name.
    * <p> Requirement:      3.2.EC.1.1 (character naming)
    * @param    nameP       Printable name for the character.
    */
    protected EncounterCharacter( String nameP )
    {   this( nameP, null );
    }

    /** Construct a new character using the given name and image file.
    * <p> Requirement:      3.2.EC.1.1 (character naming)
    * @param    nameP       Printable name for the character.
    * @param    imageFileP  Filename, relative to document base,
    *                           for the character image file.
    */
    protected EncounterCharacter( String nameP, String imageFileP )
    {   this();
        setName( nameP );
        imageFileNameI = FILE_ROOT + imageFileP;
    }
    
		/* METHODS */

    /** Adjust quality vlues, normally retaining a constant total.
    * Synchronization holds qualityValueI constant even with other
    * threads running.
    * <p> Invariants: see the class invariants
    * <p> Preconditions: qualityP is in qualityTypesS[]
    *   AND qualityValueP >= 0
    *   AND qualityValueP <= the sum of the quality values
    * <p> Postconditions: qualityP has the value qualityValueP
    *   AND the remaining quality values are in the same proportion as prior to
    *       invocation, except that values less than some tolerance are zero.
    * <p> SDD: 6.1.2.1.1
    * <p> SRS: 3.2.EC.3.2: "Configurability of Encounter character quality
    * values."
    * @param    qualityP            Quality whose value is to be adjusted.
    * @param    qualityValueP       The value to set this quality to.
    */
		
    public synchronized void adjustQuality( String qualityP,
        float qualityValueP )
    {   // Value of the quality to be changed
        float qualityValueM = qualValueI[ indexOf( qualityP ) ];

        // Save the sum of the values
        float originalSumM = sumOfQualities();

        // Set the stated quality to the desired amount,
        // adjusted to the threshold value.
        setQuality( qualityP, qualityValueP );

        // If the caller adjusts the only non-zero quality value,
        // divide the adjustment amount equally among all other qualities.
        if( originalSumM  == qualityValueM )
        {
            float qualityDiffEach = (originalSumM - qualityValueP)
                / (qualityTypeS.length - 1) ;
            for( int i = 0; i < qualityTypeS.length; ++i )
                if( !qualityTypeS[i].equalsIgnoreCase( qualityP ) )
                    setQuality( qualityTypeS[i], qualityDiffEach );
        } else {
            /* Compute factor ("proportionM") by which all other qualities must change.
            * Example: if the values were 1,3,5 (i.e. sum 9) and the first
            * qualitys changed from 1 to 2, then "3" and "5" change from 8/9 of
            * thetotal to 7/9 of the total, so each should be multiplied by 7/8,
            * i.e., by (9-2)/(9-1).
            */
            float proportionM = (originalSumM - qualityValueP) /
                (originalSumM - qualityValueM);

            //adjust the remaining qualities, retaining their mutual proportion
            for( int i = 0; i < qualityTypeS.length; ++i )
                if( !qualityTypeS[i].equalsIgnoreCase( qualityP ) )
                    setQuality( qualityTypeS[i], qualValueI[i] * proportionM );
        }
    }

    /** Fetch one of the preserved quality values.
    * @param    qualityP    String name of quality to report.
    * @return               Saved value of the quality.
    */
    public float getOldValue(String qualityP)
    {
        return oldValueI[ indexOf( qualityP ) ];
    }

    /**  Get a copy of the list of names of quality values.
    * @return           working copies of name strings representing qualities.
    */
    public static String[] getQualityTypes()
    {
        String [] returnListM =                         // Copy the string array.
            new String[qualityTypeS.length];

        for( int i = 0; i < qualityTypeS.length; i++ )  // Copy each string.
             returnListM[i] = new String(qualityTypeS[i]);

        return returnListM;                             // Return the copy.
    }

    /**   Returns the value of the specified quality.
    * <p>Precondition:      qualityP is a valid member of qualityTypeS[]
    * @param    qualityP    The quality we want the value for.
    * @return               The value of the specified quality.
    */
    public float getQualityValue( String qualityP )
    {
        return qualValueI[ indexOf( qualityP ) ];
    }

    /**  Quality values below this threshold are set to zero to
    * avoid having the game go on for an indeterminate amount of time.
    * <p>Requirement: e.g. 3.2.EC.1.2 (lower limit on non-zero quality values)
    * @return       Tolerance value
    */
    static final float getTolerance()
    {   return  0.5f;
    }

    /**  Returns the index of the the specified quality.
    * <p> Precondition: qualityP is in qualityTypeS[],
    *   give or take capitalization.
    * @param    qualityP    The quality we are searching for.
    * @return           The quality index.
    */
    protected static int indexOf( String qualityP )
    {
        // Default to "missing" value.
        int returnIndexM = -1;

        // Search quality name table and note the index value.
        for( int i = 0; i < qualityTypeS.length; ++i )
            if( qualityTypeS[ i ].equalsIgnoreCase( qualityP ) ) {
                returnIndexM = i;
                break;
            }

        return returnIndexM;
    }

    /**  Set default maximum number of characters in names of characters.
    * <p>Requirement:   3.2.EC.1.1 (limit on character name length)
    * @return           Maximum number of characters allowed in a character name
    */
    protected int maxNumCharsInName()
    {   return 15;
    }

    /** Preserve the current quality values. */
    public synchronized void saveQualityValues()
    {
        for ( int i = 0; i < qualValueI.length; i++ )
            oldValueI[i] = qualValueI[i];
    }

    /**  Set a quality value without regard to the values of other qualities.
    * Truncate any value below the threshold value down to zero.
    * Synchronization prevents changes to qualityValueI while other
    *   threads are using it.
    * <p>Requirements:      3.2.EC.2 (lower limit on non-zero quality values),
    * <p>Precondition:      qualityP is a valid member of qualityTypeS[]
    * <p>Postcondition:     Quality values are greater than tolerance or are 0.
    *
    * @param    qualityP    The quality to set the value of.
    * @param    valueP      The value to set the quality to.
    */
    public synchronized void setQuality( String qualityP, float valueP )
    {
        if( valueP < getTolerance() )
            qualValueI[ indexOf( qualityP ) ] = 0.0f;
        else
            qualValueI[ indexOf( qualityP ) ] = valueP;
    }

    /** Display the character
    * <p>Requirements: 2.1.2.1 (character displayed in game Area),
    * <br> SRS 3.2.PC.1 (character image selection),
    * <br> SRS 3.2.PQ.1 (character image in quality update window),
    * <br> SRS 3.2.AR.4.1 Display on entry of character player
    * @param    compP       UI component in which to draw the character
    * @param    drawP       Graphics context for doing the drawing.
    * @param    posP        Pixel coordinates within compP
    *                       for the center of the image.
    * @param    heightPixP  Desired image height, in pixels.
    * @param    faceLeftP   <tt>true</tt> if character faces left,
    *                       <tt>false</tt> if faces right.
    */
    public void showCharacter(Component compP, Graphics drawP, Point posP,
        int heightPixP, boolean faceLeftP)
    {
        if( imageFileNameI == null)
        {   // No image file name. Print the character name instead.
            drawP.setColor(Color.magenta);
            FontMetrics fm = drawP.getFontMetrics();
            // Print the name, centered at the character location.
            drawP.drawString( getName(),
                posP.x - fm.stringWidth( getName() ) / 2,
                posP.y - fm.getHeight()/2 );
        }
        else {  // File name was provided. Draw the image file.
            if ( chImageI == null )                         // Build on first use 
                chImageI = compP.getToolkit().getImage(imageFileNameI);
                
            int imageWidth = chImageI.getWidth(compP);      // Raw size of image
            int imageHeight = chImageI.getHeight(compP);
            int scaledWidth = imageWidth *                  // Isotropic scaling.
                    heightPixP / imageHeight;

            // Assume that the normal image faces left.
            // Decide whether to reverse the image.
            if( faceLeftP )
                // Draw the image as given, scaled and centered.
                drawP.drawImage( chImageI,
                    posP.x - scaledWidth/2, posP.y - heightPixP/2,
                    posP.x + scaledWidth/2, posP.y + heightPixP/2,
                    0, 0, imageWidth-1, imageHeight-1, null );
            else
                // Draw image reversed, scaled and centered.
                drawP.drawImage( chImageI,
                    posP.x + scaledWidth/2, posP.y - heightPixP/2,
                    posP.x - scaledWidth/2, posP.y + heightPixP/2,
                    0, 0, imageWidth-1, imageHeight-1, null );
        }

    }       // End of showCharacter.


    /**  Computes the sum of the quality values.
    * Synchronization makes sure that another thread won't change 
    * qualityValueI while this thread is part-way through computing the 
    * total.
    * <p> Requirements:  3.2.EC.3.2 (proportions among quality values),
    * <br> SRS 3.2.EC.3.1 Living points
    * @return       The sum of the player's qualities, a value 0 or greater.
    */
    public synchronized float sumOfQualities()
    {
        float sumM = 0.0f;

        for( int i = 0; i < qualityTypeS.length; ++i )
            sumM += qualValueI[i];

        return sumM;
    }

    /** To test this class.
    *   @param  argsP   destination of method test log,
    *                   class test log respectively
    */
    public static void main( String[] argsP )
    {
        // Default files on which to write test output & run tests.
        String methodOutputFileNameM = "methodOutput.txt";
        String classOutputFileNameM= "classOutput.txt";

        // Use defaults if input is improper.
        if( argsP != null && argsP.length == 2 )
        {   methodOutputFileNameM = argsP[0];
            classOutputFileNameM  = argsP[1];
        }

        // Test methods individually, then test class.
        try
        {   testEncounterCharacterMethods( methodOutputFileNameM );
            testEncounterCharacterClass( classOutputFileNameM );
        } catch( IOException eP )
        {   System.out.println( eP );
        }

        // Display the image test for a few seconds.
        Frame[] imageTests = {
            new testCharacterImage(
                new EncounterCharacter( "GuyWithNoImage", null ) ),
            new testCharacterImage(
                new EncounterCharacter( "Elena", "elena.gif" ) )
        };

        for( int i = 0; i < imageTests.length; i++) {
            imageTests[i].setSize(400, 250);
            imageTests[i].setLocation((i * 110 + 50) & 500,
                (i + 60 + 25) % 200);
            imageTests[i].setVisible(true);
            imageTests[i].show();
        }

        try {
            Thread.currentThread().sleep(15*1000);
        } catch( Exception exc) {
        }

        for( int i = 0; i < imageTests.length; i++)
            imageTests[i].dispose();
    }

    /**  Tests this class by executing its methods in combination.
    * @param        destinationP    Location to write results.
    * @exception    IOException     If there's a problem
    *                               opening or accessing destinationP
    */
    public static void testEncounterCharacterClass( String destinationP )
       throws IOException
    {   /* Prepare for the test */
        FileWriter outM = new FileWriter( new File( destinationP ) );
        System.out.println("\nEncounterCharacter class test results on "
            + destinationP + "\n" );

        /*
         * The following methods will be tested in sequences:
         *
         *  a.  adjustQuality( String qualityP, float qualityValueP )
         *  d.  deleteFromEncounterCharacters( EncounterCharacter encounterCharacterP )
         *  ge. EncounterCharacter getEncounterCharacter( String nameP )
         *  gq. float getQualityValue( String qualityP )
         *  gt. float getTolerance()
         *  io. int indexOf( String qualityP )
         *  ii. insertIntoEncounterCharacters( EncounterCharacter encounterCharacterP )
         *  m.  int maxNumCharsInName()
         *  sq. setQuality( String qualityP, float qualityValueP )
         *  so. float sumOfQualities()
             *
         *     The following sequences occur commonly:
         *     ge-aq-so
         *     ge-sq-a-gq
         *     . . . . .
         *     The following sequences have a high potential for defects:
         *     ge-aq-aq-gq-so
         *     . . . . .
         */
					
        /* Test C1: ge-aq-so */
        EncounterCharacter eC1M = new EncounterCharacter( "CharForTestC1" );    // method "ge"
        eC1M.adjustQuality(QUAL_STRENGTH,  40.0f );                             // aq
        TestExecution.printReportToFile( outM,
            "Class test ge-aq-so", eC1M.sumOfQualities(), 100.0f );             // so

        /* Test C2: ge-aq-aq-gq-so */
        EncounterCharacter eC2M = new EncounterCharacter( "CharForTestC2" );    // ge
        eC2M.adjustQuality(QUAL_STRENGTH,  40.0f );                             // aq
        eC2M.adjustQuality(QUAL_STAMINA,  20.9876f );                           // aq
        TestExecution.printReportToFile( outM,
            "Class test ge-aq-aq-gq-so: part 1",
            eC2M.getQualityValue( QUAL_STAMINA ),  20.9876f ); // gq

        TestExecution.printReportToFile( outM,
            "Class test ge-aq-aq-gq-so: part 2",
            eC2M.sumOfQualities(), 100.0f );                 // so

        /* INVARIANT-ORIENTED TESTS
        *   Check for the invariant  "qualValueI[i] >=0"
        *   -- after executing the sequences of methods executed above
        */
        boolean truthM = true;
        for( int i = 0; i < qualityTypeS.length; ++i )
        {   /* Set truthM false if any entry in eC1M.qualValueI not >= 0 */
            truthM = truthM && ( eC1M.qualValueI[i] >= 0.0f );
        }
        TestExecution.printReportToFile( outM,
            "Class test for the invariant 'qualValueI[i] >=0'", truthM, true );

        /* Conclude */
        outM.close();
        System.out.println( "\nClass tests of EncounterChar class concluded." );

    } // end of testEncouterCharacterClass


    /**  Tests all the methods of this class one at a time
    * @param        destinationP    Location to write results.
    * @exception    IOException     If there's a problem opening or
    *                               accessing destinationP
    */
    public static void testEncounterCharacterMethods( String destinationP )
        throws IOException
    {   /* Prepare for the test */
        FileWriter outM = new FileWriter( new File( destinationP )  );
        System.out.println( "EncounterCharacter method test results on "
            + destinationP + "\n" );

        /* Tests for getEncounterCharacter() */

        // normal character name
        EncounterCharacter eCNorM = new EncounterCharacter( "qwerty" );
        TestExecution.reportToFile( outM,
            "GetCharacter Test 1: nominal value", eCNorM.getName(), "qwerty" );

        // Character name null
        EncounterCharacter eCNullM = new EncounterCharacter( null );
        TestExecution.reportToFile( outM, "GetCharacter Test 2: null parameter",
            eCNullM.getName(), GameCharacter.DEFAULT_NAME );

        // Cahracter name too long
        String tooLongM = "1234567890123456789012345678901234567890";
        EncounterCharacter eCTooLongM =  new EncounterCharacter(tooLongM);
        TestExecution.reportToFile( outM,
            "GetCharacter Test 3: Limit parameter values, "
                + "max name len = " + eCTooLongM .maxNumCharsInName(),
            eCTooLongM.getName(),
            tooLongM.substring(0, eCTooLongM.maxNumCharsInName()) );

        // zero-length name
        EncounterCharacter eCZeroM = new EncounterCharacter( "" );
        TestExecution.reportToFile( outM,
            "GetCharacter Test 4: zero-length",
            eCZeroM .getName(), GameCharacter.DEFAULT_NAME );

        // bad characters
        EncounterCharacter eCPuncM = new EncounterCharacter( "a+b" );
        TestExecution.reportToFile( outM,
            "GetCharacter Test 5: bad char '+' ",
            eCPuncM .getName(), GameCharacter.DEFAULT_NAME );

        /* Tests for indexOf() for every valid quality name. */
        for( int i = 0; i < qualityTypeS.length; ++i )
            try { TestExecution.reportToFile( outM,
                "indexOf() Test 1." + i + ": valid name: " + qualityTypeS[i],
                indexOf(qualityTypeS[i]), i );
            } catch( Exception eP )
            {
                TestExecution.reportToFile( outM,
                    "indexOf() Test 1: valid name: compare ",
                    "indexOf('" + qualityTypeS[i] + "')",
                    "with expected " + i );
            }

        /* Tests for indexOf() for an invalid quality name. */
        try { TestExecution.reportToFile( outM,
            "indexOf() Test 2: invalid name: zorch", indexOf("zorch"), -1 );
        } catch( Exception eP )
        {
            TestExecution.reportToFile( outM,
                "indexOf() Test 2: valid name: compare ",
                "indexOf(\"zorch\")", "with expected -1" );
        }

        /* Tests for setQuality() */

        // Set up for test
        EncounterCharacter hank = new EncounterCharacter( "Hank" );

        // Nominal value
        hank.setQuality(QUAL_STRENGTH , 10.3f );
        TestExecution.reportToFile( outM,
            "setQuality() Test 1: nominal value",
            hank.getQualityValue( QUAL_STRENGTH ), 10.3f );

        // Out of range value
        hank.setQuality( QUAL_PATIENCE, -6.2f );
        TestExecution.reportToFile( outM,
            "setQuality() Test 2: nominal value",
            hank.getQualityValue(QUAL_PATIENCE ), 0.0f );

        // Value below close-to-zero threshold.
        hank.setQuality( QUAL_STAMINA, getTolerance() * 0.9f );
        TestExecution.reportToFile( outM,
            "setQuality() Test 3: value close to zero",
            hank.getQualityValue(QUAL_STAMINA), 0.0f );

        // Tests for adjustQuality().

        // Set up for test and verify: Values should be 20 each.
        EncounterCharacter harvey = new EncounterCharacter( "Harvey" );
        TestExecution.reportToFile( outM,
            "adjustQuality() test 0: verify that values add to 100",
            harvey.sumOfQualities(),  100.0f );

        // Nominal adjustment: strength 30 rest 70/4 each
        harvey.adjustQuality(QUAL_STRENGTH , 30.0f );
        TestExecution.reportToFile( outM,
            "adjustQuality() test 1: values sum to 100 after adjusting",
            harvey.sumOfQualities(), 100.0f );
        TestExecution.reportToFile ( outM,
            "adjustQuality() test 2: values adjusted as commanded",
            harvey.getQualityValue(QUAL_STRENGTH ), 30.0f );

        // Adjustment resulting in a zero value
        harvey.adjustQuality( QUAL_STAMINA, 99.0f );
        TestExecution.reportToFile( outM,
            "adjustQuality() test 3: verify low value reverts to zero",
            harvey.getQualityValue( QUAL_STRENGTH ), 0.0f );

        // Conclude
        outM.close();
        System.out.println( "\nMethod tests of EncounterCharacter concluded." );
    }

    /** Class to test repainting of characters. Creates a window, which
    * will contain several copies of the character image.
    */
    private static class testCharacterImage extends Frame
    {
        private EncounterCharacter characterI;

        testCharacterImage(EncounterCharacter characterP)
        {
            super(characterP.getName());
            characterI = characterP;
        }

        public void paint(Graphics drawP)
        {
            Dimension frameSizeM = getSize();
            int widthUnitM = frameSizeM.width / 5;
            int heightUnitM = frameSizeM.height / 5;

            characterI.showCharacter(this, drawP,
                new Point(widthUnitM, heightUnitM),
                heightUnitM, false);

            characterI.showCharacter(this, drawP,
                new Point(widthUnitM*4, heightUnitM*3),
                heightUnitM*2, true);

            characterI.showCharacter(this, drawP,
                new Point(widthUnitM*2, heightUnitM*2),
                heightUnitM*2, false);

            characterI.showCharacter(this, drawP,
                new Point(widthUnitM*3, heightUnitM*4),
                heightUnitM, true);
        }
    }
} // end of EncounterCharacter
