package semorg.company;

import java.sql.ResultSet;
import java.sql.SQLException;

import semorg.classes.AddressT;
import semorg.classes.Ancestor;
import semorg.classes.ContactT;
import semorg.classes.DateT;
import semorg.person.ClientListT;
import semorg.person.ClientT;

/**
 * Objects of this class are representing companies in the database.
 *
 * @author Ivan Pribela
 * @version 1.2
 */

public class CompanyT extends Ancestor
{

  /**
   * Holds information about the company.
   */

  private String shortName;

  /**
   * Holds information about the company.
   */

  private String name;

  /**
   * Holds information about the company.
   */

  private ContactT contact;

  /**
   * Holds information about the company.
   */

  private Long contactPerson;

  /**
   * Holds information about the company.
   */

  private String sector;

  /**
   * Holds information about the company.
   */

  private String function;

  /**
   * Holds information about the company.
   */

  private String shortInformation;

  /**
   * Holds information about the company.
   */

  private String notices;

  /**
   * Holds information about the company.
   */

  private float exchange;

  /**
   * Holds information about the company.
   */

  private DateT clientSince;

  /**
   * The list of clients.
   */

  private ClientListT clientList;

  /**
   * Constructs a new instace with default values for all fields.
   */

  public CompanyT()
  {
    shortName = "";
    name = "";
    contact = new ContactT();
    contactPerson = null;
    sector = "";
    function = "";
    shortInformation = "";
    notices = "";
    exchange = 0.0f;
    clientSince = new DateT();
  }

  /**
   * Constructs a new instace with the given field values.
   */

  public CompanyT(String shortName, String name, ContactT contact,
                  Long contactPerson, String sector, String function,
                  String shortInformation, String notices, float exchange,
                  DateT clientSince)
  {
    this.shortName = shortName;
    this.name = name;
    this.contact = contact;
    this.contactPerson = contactPerson;
    this.sector = sector;
    this.function = function;
    this.shortInformation = shortInformation;
    this.notices = notices;
    this.exchange = exchange;
    this.clientSince = clientSince;
  }

  /**
   * Constructs a new instace from the database.
   *
   * @param SQL a <code>ResultSet</code> instance holdin the information from
   *            the database.
   * @param column starting column from which to read data.
   */

  public CompanyT(ResultSet SQL, int Column)
  {
    this();
    try
    {
      shortName = SQL.getString(Column); Column++;
      name = SQL.getString(Column); Column++;
      contact = new ContactT(SQL, Column); Column += ContactT.noOfSQLColumns();
      contactPerson = new Long(SQL.getLong(Column)); Column ++;
      sector = SQL.getString(Column); Column++;
      function = SQL.getString(Column); Column++;
      shortInformation = SQL.getString(Column); Column++;
      notices = SQL.getString(Column); Column++;
      exchange = SQL.getFloat(Column); Column++;
      clientSince = new DateT(SQL, Column); Column += DateT.noOfSQLColumns();
    }
    catch (SQLException e)
    {
    }
  }

  /**
   * Returns the number of columns of the apropriate table in the database.
   *
   * @return the number of columns of the apropriate table in the database.
   */

  public static int noOfSQLColumns()
  {
    return 8 + ContactT.noOfSQLColumns() + DateT.noOfSQLColumns();
  }

  /**
   * Returns string representation that is used for creating the corresponding
   * table in the database.
   *
   * @param prefix prefix string added to all database column names.
   * @return string representation that is used for creating the corresponding
   *         table in the database.
   */

  public static String SQLCreateString(String prefix)
  {
    return prefix + "ShortName varchar(10), " +
           prefix + "Name varchar(50), " +
           ContactT.SQLCreateString(prefix) + ", " +
           prefix + "ContactPerson int, " +
           prefix + "Sector varchar(30), " +
           prefix + "Function varchar(30), " +
           prefix + "ShortInformation varchar(200), " +
           prefix + "Notices varchar(200), " +
           prefix + "Exchange float, " +
           DateT.SQLCreateString(prefix + "ClientSince") + ", " +
           prefix + "CONSTRAINT PK_Company PRIMARY KEY (ShortName)";
  }

  public String toSQLAddString()
  {
    return "'" + getShortName() + "', " +
           "'" + getName() + "', " +
           getContact().toSQLAddString() + ", " +
           ((this.contactPerson != null) ? this.contactPerson.toString() : "NULL") + ", " +
           "'" + getSector() + "', " +
           "'" + getFunction() + "', " +
           "'" + getShortInformation() + "', " +
           "'" + getNotices() + "', " +
           getExchange() + ",  " +
           getClientSince().toSQLAddString();
  }

  public String toSQLModifyString(String prefix)
  {
    return prefix + "ShortName = '" + getShortName() + "', " +
           prefix + "Name = '" + getName() + "', " +
           getContact().toSQLModifyString(prefix) + ", " +
           prefix + "ContactPerson = " + ((this.contactPerson != null) ? this.contactPerson.toString() : "NULL") + ", " +
           prefix + "Sector = '" + getSector() + "', " +
           prefix + "Function = '" + getFunction() + "', " +
           prefix + "ShortInformation = '" + getShortInformation() + "', " +
           prefix + "Notices = '" + getNotices() + "', " +
           prefix + "Exchange = " + getExchange() + ", " +
           getClientSince().toSQLModifyString(prefix + "ClientSince");
  }

  public String toString()
  {
    return name;
  }

  /**
   * Returns short name of a company.
   *
   * @return short name of a company.
   */

  public String getShortName()
  {
    return shortName;
  }

  /**
   * Returns name of a company.
   *
   * @return name of a company.
   */

  public String getName()
  {
    return name;
  }

  /**
   * Returns address of a company.
   *
   * @return address of a company.
   */

  public AddressT getAddress()
  {
    return contact.getAddress();
  }

  /**
   * Returns contact information of a company.
   *
   * @return contact information of a company.
   */

  public ContactT getContact()
  {
    return contact;
  }

  /**
   * Returns contact person information of a company.
   *
   * @return contact person information of a company.
   */

  public ClientT getContactPerson()
  {
    if (clientList == null)
    {
      return null;
    }
    else if (contactPerson == null)
    {
      return null;
    }
    else
    {
      return (ClientT) clientList.find(contactPerson);
    }
  }

  /**
   * Returns sector of a company.
   *
   * @return sector of a company.
   */

  public String getSector()
  {
    return sector;
  }

  /**
   * Returns function of a company.
   *
   * @return function of a company.
   */

  public String getFunction()
  {
    return function;
  }

  /**
   * Returns short information of a company.
   *
   * @return short information of a company.
   */

  public String getShortInformation()
  {
    return shortInformation;
  }

  /**
   * Returns notices on the company.
   *
   * @return notices on the company.
   */

  public String getNotices()
  {
    return notices;
  }

  /**
   * Returns exchange rate for the company.
   *
   * @return exchange rate for the company.
   */

  public float getExchange()
  {
    return exchange;
  }

  /**
   * Returns the date when this company become a client.
   *
   * @return the date when this company become a client.
   */

  public DateT getClientSince()
  {
    return clientSince;
  }

  /**
   * Sets short name of the company.
   *
   * @param value new value.
   */

  public void setShortName(String value)
  {
    shortName = value;
  }

  /**
   * Sets name of the company.
   *
   * @param value new value.
   */

  public void setName(String value)
  {
    name = value;
  }

  /**
   * Sets address name of the company.
   *
   * @param value new value.
   */

  public void setAddress(AddressT value)
  {
    contact.setAddress(value);
  }

  /**
   * Sets contact information for the company.
   *
   * @param value new value.
   */

  public void setContact(ContactT value)
  {
    contact = value;
  }

  /**
   * Sets contact person of the company.
   *
   * @param value new value.
   */

  public void setContactPerson(ClientT value)
  {
    if (value != null)
    {
      contactPerson = new Long(value.getIDNumber());
    }
    else
    {
      contactPerson = null;
    }
  }

  /**
   * Sets contact person name of the company.
   *
   * @param value new value.
   */

  public void setContactPerson(Long value)
  {
    contactPerson = value;
  }

  /**
   * Sets sector of the company.
   *
   * @param value new value.
   */

  public void setSector(String value)
  {
    sector = value;
  }

  /**
   * Sets function of the company.
   *
   * @param value new value.
   */

  public void setFunction(String value)
  {
    function = value;
  }

  /**
   * Sets short information of the company.
   *
   * @param value new value.
   */

  public void setShortInformation(String value)
  {
    shortInformation = value;
  }

  /**
   * Sets notices on the company.
   *
   * @param value new value.
   */

  public void setNotices(String value)
  {
    notices = value;
  }

  /**
   * Sets exchange rate for the company.
   *
   * @param value new value.
   */

  public void setExchange(float value)
  {
    exchange = value;
  }

  /**
   * Sets the date since this company become a client.
   *
   * @param value new value.
   */

  public void setClientSince(DateT value)
  {
    clientSince = value;
  }

  /**
   * Sets client list for this booking.
   *
   * @param value client list for this booking
   */

  public void setClientList(ClientListT value)
  {
    this.clientList = value;
  }

  public Object getKey()
  {
    return getShortName();
  }

}
