package semorg.classes;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;

/**
 * This class is ancestor of all classes that represent a list of elements
 * stored in database.
 *
 * @author Ivan Pribela
 * @version 1.0
 */

public abstract class List extends Object
{

  /**
   * Associated database.
   */

  protected DataBaseT database;

  /**
   * Construct fresh instance associated with the given database.
   * 
   * @param database the database to be associated with this list.
   */

  protected List(DataBaseT database)
  {
    this.database = database;
  }

  /**
   * Construct fresh instance associated with desired database through the
   * given <code>Connection</code> object.
   * 
   * @param DBConnection connection with the data base.
   * @param database the database to be associated with this list.
   */

  protected List(Connection DBConnection)
  {
    this(new DataBaseT(DBConnection));
  }

  /**
   * Creates a new table in associated database. If a table with appropriate
   * name allready exists operation fails.
   * 
   * @return <code>true</code> if operation was successfull; <code>false</code>
   *         if not.
   */

  public abstract boolean create();

  /**
   * Creates a new table in associated database. If a table with appropriate
   * name allready exists operation will fail.
   * 
   * @param tableName name of table to be created.
   * @param createString string used to describe the created table. 
   * @return <code>true</code> if operation was successfull; <code>false</code> 
   *         if not.
   */

  protected boolean create(String tableName, String createString)
  {
    return database.update("CREATE TABLE " + tableName + " " +
                           "(" + createString + ")");
  }

  /**
   * Destroyes the table in associated database.
   * 
   * @return <code>true</code> if operation was successfull; <code>false</code> 
   *         if not.
   */

  public abstract boolean destroy();

  /**
   * Destroyes the table in associated database.
   * 
   * @param tableName name of table to be destroyed.
   * @return <code>true</code> if operation was successfull; <code>false</code> 
   *         if not.
   */

  protected boolean destroy(String tableName)
  {
    return database.update("DROP TABLE " + tableName);
  }

  /**
   * Adds one element to the database.
   * 
   * @param element element to be added to the database.
   * @return <code>true</code> if operation was successfull; <code>false</code> 
   *         if not.
   */

  public abstract boolean add(Ancestor element);

  /**
   * Adds one element to the database.
   * 
   * @param tableName name of the database table.
   * @param tableColumns columns of the database table.
   * @param element element to be added to the database.
   * @return <code>true</code> if operation was successfull; <code>false</code> 
   *         if not.
   */

  protected boolean add(String tableName, String tableColumns, String element)
  {
    return database.update("INSERT INTO " + tableName +
                           " (" + tableColumns + ") " +
                           "VALUES (" + element + ")");
  }

  /**
   * Removes one element from the database.
   * 
   * @param key primary key of element to remove.
   * @return <code>true</code> if operation was successfull; <code>false</code> 
   *         if not.
   */

  public abstract boolean remove(Object key);

  /**
   * Removes one element from the database.
   * 
   * @param tableName name of table.
   * @param tableKey name of a primary key column.
   * @param key primary key of element to remove.
   * @return <code>true</code> if operation was successfull; <code>false</code> 
   *         if not.
   */

  protected boolean remove(String tableName, String tableKey, Object key)
  {
    return database.update("DELETE FROM " + tableName + " " +
                           "WHERE (" + tableKey + " = " + stringRepresentation(key) + ")");
  }

  /**
   * Removes one element from the database.
   * 
   * @param tableName name of table.
   * @param tableKey1 name of a part of primary key column.
   * @param key1 part of primary key of element to remove.
   * @param tableKey2 name of a part of primary key column.
   * @param key2 part of primary key of element to remove.
   * @return <code>true</code> if operation was successfull; <code>false</code> 
   *         if not.
   */

  protected boolean remove(String tableName, String tableKey1, Object key1, String tableKey2, Object key2)
  {
    return database.update("DELETE FROM " + tableName + " " +
                           "WHERE (" + tableKey1 + " = " + stringRepresentation(key1) +
                           " AND " + tableKey2 + " = " + stringRepresentation(key2) + ")");
  }

  /**
   * Updates data about the given element in the database.
   * 
   * @param element element to be updated.
   * @return <code>true</code> if operation was successfull; <code>false</code> 
   *         if not.
   */

  public abstract boolean modify(Ancestor element);

  /**
   * Updates data about the given element in the database.
   * 
   * @param tableName name of the database table.
   * @param tableKey name of a primary key column.
   * @param modifyString element to be updated.
   * @param key primary key of element to modify.
   * @return <code>true</code> if operation was successfull; <code>false</code> 
   *         if not.
   */

  protected boolean modify(String tableName, String tableKey, String modifyString, Object key)
  {
    return database.update("UPDATE " + tableName + " " +
                           "SET " + modifyString + " " +
                           "WHERE (" + tableKey + " = " + stringRepresentation(key) + ")");
  }

  /**
   * Updates data about the given element in the database.
   * 
   * @param tableName name of the database table.
   * @param tableKey1 name of a part of primary key column.
   * @param tableKey2 name of a part of primary key column.
   * @param modifyString element to be updated.
   * @param key1 part of primary key of element to remove.
   * @param key2 part of primary key of element to remove.
   * @return <code>true</code> if operation was successfull; <code>false</code> 
   *         if not.
   */

  protected boolean modify(String tableName, String tableKey1, String tableKey2, String modifyString, Object key1, Object key2)
  {
    return database.update("UPDATE " + tableName + " " +
                           "SET " + modifyString + " " +
                           "WHERE (" + tableKey1 + " = " + stringRepresentation(key1) +
                           " AND " + tableKey2 + " = " + stringRepresentation(key2) + ")");
  }

   
  /**
   * Returns the number of elements contained in the database.
   * 
   * @return the number of elements contained in the database.
   */

  public abstract int count();

  /**
   * Returns the number of elements contained in the database.
   * 
   * @param tableName name of table.
   * @return the number of elements contained in the database.
   */

  protected int count(String tableName)
  {
    ResultSet Result = database.query("SELECT COUNT(*) " +
                                      "FROM " + tableName);
    if (Result == null)
    {
      return 0;
    }
    else
    {
      int noElements = 0;
      try
      {
        if (Result.next())
        {
          noElements = Result.getInt(1);
        }
        else
        {
          noElements = 0;
        };
      }
      catch (SQLException e)
      {
        noElements = 0;
      }
      database.close(Result);
      return noElements;
    }
  }

  /**
   * Retreves one element from the database.
   * 
   * @param index index of the element to be retrieved.
   * @return requested element.
   */

  public abstract Ancestor get(int index);

  /**
   * Retreves one element from the database.
   * 
   * @param tableName name of table.
   * @param index index of the element to be retrieved.
   * @return requested element.
   */

  protected ResultSet get(String tableName, int index)
  {

    // Index too small
    if (index < 0)
    {
      return null;
    }

    // Get data from database
    return database.query("SELECT * " +
                          "FROM " + tableName);
                          
  }
  
  
  
  
  /**
   * Searches the database for element with the given primary key.
   * 
   * @param key primaru key of element to be found.
   * @return data that has been found.
   */

  public abstract Ancestor find(Object key);

  /**
   * Searches the database for element with the given primary key.
   * 
   * @param tableName name of table.
   * @param tableKey name of a part of primary key column.
   * @param key primaru key of element to be found.
   * @return data that has been found.
   */

  public ResultSet find(String tableName, String tableKey, Object key)
  {
    return database.query("SELECT * " +
                          "FROM " + tableName + " " +
                          "WHERE (" + tableKey + " = " + stringRepresentation(key) + ")");
  }

  /**
   * Searches the database for element with the given primary key.
   * 
   * @param tableName name of table.
   * @param tableKey1 name of a part of primary key column.
   * @param key1 part of primary key of element to remove.
   * @param tableKey2 name of a part of primary key column.
   * @param key2 part of primary key of element to remove.
   * @return data that has been found.
   */
  
  
  protected ResultSet find(String tableName, String tableKey1, Object key1, String tableKey2, Object key2)
  {
    return database.query("SELECT * " +
                          "FROM " + tableName + " " +
                          "WHERE (" + tableKey1 + " = " + stringRepresentation(key1) +
                          " AND " + tableKey2 + " = " + stringRepresentation(key2) + ")");
  }
  
  
  

  /**
   * Returns a String object representing this list.
   *
   * @return a string representation of this list.
   */

  public String toString()
  {
    return "";
  }

  /**
   * Returns string representation of the given object.
   * 
   * @param object object which string representation is requested.
   * @return string representation of the given object.
   */

  protected String stringRepresentation(Object object)
  {
    if (object instanceof String)
    {
      return "\'" + object.toString() + "\'";
    }
    else
    {
      return object.toString();
    }
  }
  
  public ResultSet get(String tableName, String uslov1, Object key1, String uslov3) {
  	 // Get data from database
    return database.query("SELECT * FROM " + tableName +
                          " WHERE (" + uslov1 + " " + uslov3  +" " + stringRepresentation(key1) + ")");
  }
   
}
