package semorg.sql.util;

import semorg.gui.util.Messages;

/**
 * Models a condition of a WHERE-clause, which is structured as follows<br>
 * <br>
 * <code> < columnName > < relation > < value > < conjunction >,</code><br>
 * <br>
 * whereas:
 * <ul>
 * <li>columnName = attribute of relation (table)</li>
 * <li>relation = <, >, !=, =, ~</li>
 * <li>value = a value (e.g. 1,'Schmidt', ...)</li>
 * <li>conjunction = AND-, OR- or no conjunction</li>
 * </ul>
 */
public class DBConstraint {
    /** Equality relation. */
    public static final int REL_EQ = 0;

    /** Inequality relation. */
    public static final int REL_NEQ = 1;

    /** "Less Than" relation. */
    public static final int REL_LESS = 2;

    /** "Greater Than" relation. */
    public static final int REL_GREATER = 3;

    /** LIKE relation. */
    public static final int REL_LIKE = 4;

    /** This conjunction type marks the end of chain of conjunctions. */
    public static final int CONJ_END = 0;

    /** AND conjunction. */
    public static final int CONJ_AND = 1;

    /** OR conjunction. */
    public static final int CONJ_OR = 2;

    /** The name of the column which is restricted by the conjunction. */
    private String columnName;

    /** Relation (<, >, !=, =, ~) between attribute and value. */
    private int relation;

    /** The value, which restricts the attribute. */
    private Object value;

    /** Conjunction, which appears AFTER the condition (possibly none). */
    private int conjunction;

    /**
     * Creates a {@link DBConstraint} instance with the given values.
     * 
     * @param columnName
     *                name of the attribute which should be restricted.
     * @param relation
     *                Relation between attribute and value.
     * @param value
     *                Value to restrict the attribute.
     * @param conjunction
     *                Conjunction, which appears AFTER the condition.
     */
    public DBConstraint(String columnName, int relation, Object value,
	    int conjunction) {
	this.columnName = columnName;
	this.relation = relation;
	this.value = value;
	this.conjunction = conjunction;
    }

    public String getColumnName() {
	return columnName;
    }

    public int getConjunction() {
	return conjunction;
    }

    public int getRelation() {
	return relation;
    }

    public Object getValue() {
	return value;
    }

    /**
     * Returns for a given relation string the appropriate relation constant.
     * 
     * @param relationString
     *                a string, which contains one of the operators <,>,~,!=, =.
     */
    public static int getRelation(String relationString) {
	if (relationString.equals("<"))
	    return REL_LESS;
	else if (relationString.equals(">"))
	    return REL_GREATER;
	else if (relationString.equals("~"))
	    return REL_LIKE;
	else if (relationString.equals("!="))
	    return REL_NEQ;
	else
	    return REL_EQ;
    }

    /**
     * Return for given conjunction string the appropriate conjunction constant.
     * 
     * @param conjunctionString
     *                String containing a conjunction like "and" / "or".
     */
    public static int getConjunction(String conjunctionString) {
	if (conjunctionString.equals(Messages
		.getString("DBConstraint.AndConjunction")))
	    return CONJ_AND;
	else if (conjunctionString.equals(Messages
		.getString("DBConstraint.OrConjunction")))
	    return CONJ_OR;
	else
	    return CONJ_END;
    }
}
